/*
 * Program to compress Megha format data. Output consists of a stream of
 * bits packed into 32 bit words with the first bit being the LSB of the
 * word. The data stream consists of:
 *
 * 9 bits of channel information
 * 13 bits of timing information for the channel
 * 13 bits of energy information for the channel
 *
 * The channel information is obtained by taking the raw channel and dividing
 * by 2.
 *
 * The end of an event is signified by channel = 511, timing = 8191, energy =
 * 8191. A half filled block is signified by channel = 511, timing = 8190,
 * energy = 8190.
 */

#include <stdio.h>
#include "inttypes.h" /* From here or from /usr/include */
#include <math.h>
#include "megha.h"

int writebits(uint32_t data, int numbits)
{
    static uint32_t buffer[4096], *bp = buffer;
    static uint32_t *be = buffer + sizeof(buffer)/sizeof(uint32_t);
    static unsigned int c = 0;

    data &= (1<<numbits) - 1;

    if (numbits < 0)
    {
	if (c != 0 || bp != buffer)
	    if (errwrite(1, (char *) buffer, sizeof(buffer)))
		return -1;
	bp = buffer;
	c = 0;
    }
    else if (c + numbits > 32)
    {
	*bp++ |= data << c;
	if (bp == be)
	{
	    if (errwrite(1, (char *) buffer, sizeof(buffer)))
		return -1;
	    bp = buffer;
	}
	*bp = data >> (32 - c);
	c = numbits + c - 32;
    }
    else
    {
	*bp |= data << c;
	c += numbits;
	if (c == 32)
	{
	    bp++;
	    if (bp == be)
	    {
		if (errwrite(1, (char *) buffer, sizeof(buffer)))
		    return -1;
		bp = buffer;
	    }
	    *bp = 0;
	    c = 0;
	}
    }

    return 0;
}

int writeout(uint32_t channel, uint32_t time, uint32_t energy)
{
    static int channelbits = 0, timingbits, energybits;

    if (channelbits == 0)
    {
	channelbits = CHANNELBITS;
	timingbits = TIMINGBITS;
	energybits = ENERGYBITS;

	if (writebits(channelbits, 4))
	    return -1;
	if (writebits(timingbits, 4))
	    return -1;
	if (writebits(energybits, 4))
	    return -1;
    }

    if (channel == CONTROLCHANNEL)
    {
	if (writebits((1<<channelbits)-1, channelbits))
	    return -1;
	if (writebits(time, CONTROLBITS))
	    return -1;
	if (time == CONTROLEXTENDED)
	{
	    if (writebits(energy, EXTENDEDBITS))
		return -1;
	    if (energy == EXTENDEDEOB)
	    {
		channelbits = 0;
		return writebits(0, -1);
	    }
	}
	return 0;
    }

    if (channel >= (1<<channelbits)-1)
    {
	writebits((1<<channelbits)-1, channelbits);
	writebits(CONTROLEXTENDED, CONTROLBITS);
	writebits(EXTENDEDCBITS, EXTENDEDBITS);
	while((1<<channelbits) <= channel+1)
	    channelbits++;
	writebits(channelbits, 4);
	fprintf(stderr, "Channel range extended to %d (%d bits).\n",
		(1<<channelbits)-2, channelbits);
    }
	
    if (time > (1<<timingbits)-1)
    {
	writebits((1<<channelbits)-1, channelbits);
	writebits(CONTROLEXTENDED, CONTROLBITS);
	writebits(EXTENDEDTBITS, EXTENDEDBITS);
	while((1<<timingbits) <= time)
	    timingbits++;
	writebits(timingbits, 4);
	fprintf(stderr, "Time range extended to %d (%d bits).\n",
		(1<<timingbits)-1, timingbits);
    }
	
    if (energy > (1<<energybits)-1)
    {
	writebits((1<<channelbits)-1, channelbits);
	writebits(CONTROLEXTENDED, CONTROLBITS);
	writebits(EXTENDEDEBITS, EXTENDEDBITS);
	while((1<<energybits) <= energy)
	    energybits++;
	writebits(energybits, 4);
	fprintf(stderr, "Energy range extended to %d (%d bits).\n",
		(1<<energybits)-1, energybits);
    }
	
    if (writebits(channel, channelbits))
	return -1;
    if (writebits(time, timingbits))
	return -1;
    return writebits(energy, energybits);
}

static const char *nomoreerrors = "No more errors of this type will be "
    "reported.\n";

int main(int argc, const char *const *argv)
{
    unsigned short buffer[8192], *sp, *ep;
    int len, block, lastchan, time, energy;
    int badstart = 0, badlength = 0, badshort = 0, badlong = 0, badpair = 0;
    int badrepeat = 0, badend = 0;
    int warning = 10;
    const char *opt;

    argc--;
    argv++;

    while(argc)
    {
	opt = *argv++;
	argc--;
	if (*opt == '-')
	{
	    opt++;
	    if (*opt == 'w')
	    {
		opt++;
		if (*opt == '\0')
		{
		    if (argc == 0)
		    {
			fprintf(stderr, "-w expects to be followed by a "
				"number.\n");
			return 1;
		    }
		    opt = *argv++;
		    argc--;
		}
		warning = atoi(opt);
	    }
	    else
	    {
		fprintf(stderr, "Unrecognised option '%c'.\n", *opt);
		return 1;
	    }
	}
	else
	{
	    fprintf(stderr, "Unexpected parameter \"%s\", use < to specify "
		    "the input file.\n", opt);
	    return 1;
	}
    }

    for(block = 0; ; block++)
    {
	if (errread(0, (char *) buffer, sizeof(buffer)))
	    break;

	sp = buffer;

/*
	for(len = 0; len < 16; len++)
	    fprintf(stderr, "%04x ", sp[len]);
	putc('\n', stderr);
*/

	if (*((unsigned int *) sp) == 0x4d454748 &&
	    ((unsigned int *) sp)[1] == 0x41444154)
	    sp += 12;
	else if (*((unsigned int *) sp) == 0x45474556 &&
	    ((unsigned int *) sp)[1] == 0x454e5444)
	    sp += 12;

	for(;;)
	{
	    ep = sp;

	    if (*ep != 0xffff)
	    {
		if (badstart < warning)
		    fprintf(stderr, "Position %d/%d: Event does not start "
			    "with 0xffff, ", block, ep - buffer);
		badstart++;
		ep++;
		do {
		    ep++;
		} while(ep < buffer + 8192 && *ep != 0xffff);
		if (ep == buffer + 8192)
		{
		    if (badstart <= warning)
			fprintf(stderr, "skipping record.\n");
		    if (badstart == warning)
			fputs(nomoreerrors, stderr);
		    break;
		}
		if (badstart <= warning)
		    fprintf(stderr, "recovering at %d.\n", ep - buffer);
		if (badstart == warning)
		    fputs(nomoreerrors, stderr);
		if ((ep - buffer) & 1)
		{
		    uint32_t *ip;

		    for(ip = (uint32_t *) buffer;
			ip < (uint32_t *) (buffer + 8192); ip++)
			*ip = (*ip>>16) | (ip[1]<<16);
		    ep -= 3;
		}
		sp = ep;
	    }

	    if ((len = ep[1]) > 1024 || ep + len >= buffer + 8190)
	    {
		if (badlength < warning)
		    fprintf(stderr, "Position %d/%d: Suspicious event length, "
			    "skipping record.\n", block, ep - buffer);
		badlength++;
		if (badlength == warning)
		    fputs(nomoreerrors, stderr);
		break;
	    }

	    if (len == 0)
		break;

	    sp += 2;
	    for(lastchan = -1; sp < ep + len; sp++)
	    {
		if (*sp == 0xffff)
		{
		    if (badshort < warning)
			fprintf(stderr, "Position %d/%d: Event too short "
				"(%d<%d), skipping event\n", block,
				ep - buffer, sp - ep, len);
		    badshort++;
		    if (badshort == warning)
			fputs(nomoreerrors, stderr);
		    break;
		}
		if ((*sp & 0xe000) == 0x2000)
		{
		    if ((sp[1] & 0xe000) == 0x8000)
		    {
			if (*sp & 1)
			    energy = sp[1] & 0x1fff;
			else
			    time = sp[1] & 0x1fff;
			if (lastchan == -1)
			    lastchan = *sp & 0x1fff;
			else
			{
			    if ((lastchan & 0x1ffe) != (*sp & 0x1ffe))
			    {
				if (badpair < warning)
				    fprintf(stderr, "Position %d/%d: %d and "
					    "%d not paired, skipping "
					    "channel.\n", block, sp - buffer,
					    lastchan, *sp & 0x1fff);
				badpair++;
				if (badpair == warning)
				    fputs(nomoreerrors, stderr);
				lastchan = *sp & 0x1fff;
			    }
			    else if ((*sp & 0x1fff) == lastchan)
			    {
				if (badrepeat < warning)
				    fprintf(stderr, "Position %d/%d: channel "
					    "%d repeated, taking second.\n",
					    block, sp - buffer, lastchan);
				badrepeat++;
				if (badrepeat == warning)
				    fputs(nomoreerrors, stderr);
			    }
			    else
			    {
				if (writeout(lastchan>>1, time, energy))
				   return 1;
				lastchan = -1;
			    }
			}
			sp++;
		    }
		}
	    }
	    
	    if (lastchan != -1)
	    {
		if (badpair < warning)
		    fprintf(stderr, "Position %d/%d: No pair for channel %d, "
			    "skipping channel.\n", block, sp - buffer,
			    lastchan);
		badpair++;
		if (badpair == warning)
		    fputs(nomoreerrors, stderr);
	    }

	    if (writeout(CONTROLCHANNEL, CONTROLEOE, 0))
		return 1;

	    if (*sp != 0xffff)
	    {
		while(*sp != 0xffff && sp < buffer + 8192)
		    sp++;
		if (sp < buffer + 8192)
		{
		    if (badlong < warning)
			fprintf(stderr, "Position %d/%d: Event too long "
				"(%d>%d), skipping event.\n", block,
				ep - buffer, sp - ep, len);
		    badlong++;
		    if (badlong == warning)
			fputs(nomoreerrors, stderr);
		}
		else
		{
		    if (badend < warning)
			fprintf(stderr, "Position %d/%d: Missing event end, "
				"skipping record\n", block, ep - buffer);
		    badend++;
		    if (badend == warning)
			fputs(nomoreerrors, stderr);
		    break;
		}
	    }
	}
    }

    if (writeout(CONTROLCHANNEL, CONTROLEXTENDED, EXTENDEDEOB))
	return 1;
    
    errstatus();
    fprintf(stderr, "Blocks with bad start           : %d\n", badstart);
    fprintf(stderr, "Events running off end of block : %d\n", badend);
    fprintf(stderr, "Events with bad length          : %d\n", badlength);
    fprintf(stderr, "Events shorter than expected    : %d\n", badshort);
    fprintf(stderr, "Events longer than expected     : %d\n", badlong);
    fprintf(stderr, "Channels not paired             : %d\n", badpair);
    fprintf(stderr, "Channels repepated in event     : %d\n", badrepeat);

    return 0;
}
