

/*
   This program reads ANSI format tapes into Unix files.
   Copyright 1981 by Dave Phillips, all rights reserved. 
*/

/* Author: Dave Phillips 		 */
/* University of Central Florida	 */
/* duke!ucf-cs!phillips			 */

/* Modified by Mike Mitchell		 */
/* North Carolina State University	 */
/* duke!mcnc!ncsu!mcm			 */
/*					 */
/* added c, r, P, F, and S options.	 */
/* vmstp now writes VMS format tapes.	 */
/* Last modified 11/2/83 by Mike Mitchell */
/* now writes 2048 byte blocks           */
/* default device now rst0 for exabyte   */
/* physical mode (p) added for sort file */
/* fix to allow 2,3 for tape number      */
/* now write 8192 byte blocks		 */
/* and p mode also writing sort file	 */
/* fix to F and U format for T,t,v       */
/* prev modified 01/04/92 J.Campbell     */
/* last modified 15/12/95                */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include <regexpr.h>
#include <alloca.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>

#include "ansitp.h"

#define SVR4

char labelbuf[LABELSIZE+4];	/* i/o buffer for tape labels */
char *iobuf  = NULL;	/* pointer to main i/o buffer */
char *Ibuf = NULL;	/* pointer to input buffer (for writing tape) */
int  bufsize = 0;	/* current size of iobuf */

static ansitp_info_t in, out;      /* ansitp current info about tape volume */

char **namelist;	/* ptr to an array of string pointers (output) */
FLAG *nameflag;		/* ptr to a array of flags */
int namecnt = 0;	/* number of names in namelist */
FLAG gotregex = 0;	/* flag if have any regular expressions */

/* data for use with disk files */
static char	*outdir	 = NULL;
FILE            *sout;	        /* output stream pointer */

FLAG	Extract, Dirflag, No_nl, Addlnum, Ucase, Ecase, Verbose;
FLAG	Writefile, Lineno, Strip, Duplicate;
FLAG	Fixed, Newtape, Oldtape, Padd, Fnl, Physical, Scantape;

static void fix2gblimit(int fd);
int prtdir(ansitp_info_t *cur, FLAG flag);
void prt_labels(ansitp_info_t *cur);

int
main(int argc, char **argv)
{
	int r;
	char *progname = argv[0];
	extern char *optarg;
	extern int optind;

	if (argc < 2)
	      usage();

/* set default tape streams */	
	out.device = in.device = "/dev/rmt/0";

/* skip over program name */
	argv[argc] = 0;
	progname = *argv++;
	argc--;

/* get main command key -- only one allowed */
	switch(**argv) {
		  case 'x':			/* Extract listed files from tape */
		    Extract++;
		    break;
		  case 't':			/* print short directory */
		    Dirflag = 1;
		    Verbose++;
		    break;
		  case 'T':			/* print long directory */
		    Dirflag = 2;
		    Verbose++;
		    break;
		  case 's':			/* quickly scan tape */
		    Scantape++;
		    Verbose++;
		    break;		
		  case 'c':			/* create a tape */
		    Newtape++;
		    Ecase++;
		    break;
		  case 'r':			/* append to end of tape */
		    Oldtape++;
		    Ecase++;
		    break;
		  case 'd':                     /* duplicate source tape on destination tape */
		    Duplicate++;
		    out.device = "/dev/rmt/1";
		    break;
		  default:
		    usage();
	      }
		    
	if (!(Extract || Dirflag || Scantape || Newtape || Oldtape || Duplicate))
	      usage();	 /* need at least one of them */

/* get other arguments */
	*argv = progname;
	while((r = getopt(argc, argv, "f:i:o:nNlUEvSFPp")) != EOF) 
	      switch(r) {
		  case 'f':                     /* input file */
		  case 'i':
		    in.device = optarg;
		    break;
		  case 'o':			/* directory to place output in */
		    if (Extract)
			  outdir = optarg;
		    else
			  out.device = optarg;
		    break;
		  case 'n':		       /* don't add new-lines to D format records */
		    No_nl++;
		    break;
		  case 'N':			/* add new-lines to F format records */
		    Fnl++;
		    break;
		  case 'l':			/* add SOS line numbers to output */
		    Addlnum++;
		    break;
		  case 'U':			/* don't lower case filenames on tape */
		    Ucase++;
		    break;
		  case 'E':			/* don't upper case filename list */
		    Ecase++;
		    break;
		  case 'v':			/* be Verbose - list all files */
		    Verbose++;                  /* if verbose > 1 dump tape labels */
		    break;
		  case 'S':			/* strip trailing blanks in F format */
		    Strip++;
		    break;
		  case 'F':			/* Fixed-length records on output */
		    Fixed++;
		    break;
		  case 'P':			/* Pad fixed-length records on output */
		    Padd++;
		    break;
		  case 'p':			/* Physical adds/strips headers in block */
		    Physical++;
		    break;
		  case '?':
		    usage();
		    break;
	      }
	

/* copy remainder of arguments into regular expression matching buffer */
	argv += optind;
	if (r = copy_regex(argv))               /* copy and convert reg expr */
	      exit(r);	
	
	if (r = process())			/* process tape */
	      exit(r);

	return(0);
}

int
usage(void)
{
      fprintf(stderr,"ansitp Usage:\nTo extract files from tape to disk:\n\t");
      fprintf(stderr,"ansitp x -[nNlUEvSFPp] [-i input tape] [-o output directory] [name ...]\n");
      fprintf(stderr,"To scan files on tape:\n\t");
      fprintf(stderr,"ansitp s|t|T [-v] [-i input tape]\n");
      fprintf(stderr,"To add disk files to a new or old tape:\n\t");
      fprintf(stderr,"ansitp c|r -[nNlUEvSFPp] [-o output tape] [name ...]\n");
      fprintf(stderr,"To copy files on input tape to output tape:\n\t");
      fprintf(stderr,"ansitp d -[v] [-i input tape] [-o output tape] [name ...]\n");
      fprintf(stderr,"\nE.G. to copy NSF EbyE data files from tape drive 1 to /stage on disk use:\n");
      fprintf(stderr,"\tansitp x -p -i /dev/rmt/1 -o /stage\n");
      return(1);
}

int
open_tape_device(ansitp_info_t *cur, int rw)
{
      if ((cur->fd = open(cur->device, rw)) < 0) {
	    fprintf(stderr,"ansitp: opening tape file %s : ",cur->device);
	    perror("");
	    return(1);
      }
      cur->open = TRUE;
      /* ensure source tape is at beginning */
      if (tapeop(cur->fd, MTREW, 1) != 0) {
	    fprintf(stderr,"ansitp: error rewinding tape %s : ",cur->device);
	    return(closetp(1));
      }
      return(0);
}

int
read_vol_header(ansitp_info_t *cur)
{
      int n = read(cur->fd,labelbuf,LABELSIZE);	/* read volume label */
      if (n < 0) {
	    perror("ansitp: reading volume label");
	    return(closetp(1));
      }
      if (n != LABELSIZE) {
	    fprintf(stderr,"ansitp: VOL1 label block wrong length!\n");
	    fprintf(stderr,"ansitp: block length is %d,image is -\n%80s\n",n,labelbuf);
	    return(closetp(1));
      }
      if (sscanf(labelbuf,"VOL1%6s",cur->volname) != 1) 
	    return( label_error("missing volume label!") );

      if (Verbose > 1) {
	    ansi_vol1_t *v = (ansi_vol1_t *) &labelbuf;
	    fprintf(stderr,"ANSI volume label from tape on device %s:\n",cur->device);
	    fprintf(stderr,"\tname = %-.6s owner = %-.14s access = '%c', level = %c\n",
		    v->label,v->owner,v->access,v->ansi_level);
      }
      else {
	    fprintf(stderr,"Using ANSI tape volume named %s on device %s\n",cur->volname,cur->device);
      }
      return(0);
}

int
process(void)
{
	int r;		/* value returned from a function */
	int writetape = (Newtape | Oldtape | Duplicate);
	int readtape  = ! (Newtape | Oldtape);

/* open tape devices */	
	if (readtape) {
	      if ( r = open_tape_device(&in, O_RDONLY))
		    return(r);
	      /* read volume header */
	      if ( r = read_vol_header(&in))
		    return(r);
	}
	if (writetape) {
	      if ( r = open_tape_device(&out, O_RDWR))
		    return(r);
	      /* read volume header */
	      if (Oldtape)
		    if ( r = read_vol_header(&out))
			  return(r);
	}

/* do the disk and tape operations */
	if (Newtape) {
	      if (r = newvol())
		    return(r);
	      return(write_tape());
	}
	else if (Oldtape) {
	      if (r = skip_tape(out.fd))
		    return(r);
	      return(write_tape());
	}
	else if (Duplicate) {
	      if (r = query_dup_tape()) 
		    return(r);
	      return( dup_tape());
	}
	else
	      return( read_tape());

} /* end process */

int
read_tape(void)
{
      int r, readable;
      int selectfile;

      for (;;) {
	    r = readhead();		/* read header labels */
	    if (r) return(r == -1 ? 0 : r);
	    
	    selectfile = (*namelist==0) || wanted(in.tapefile);
	    Writefile = selectfile && Extract;
	    
	    readable = setup();                /* setup for copying */
	    
	    if (readable && (Writefile | Dirflag))
		  r = copyfile();
	    else if (! readable)
		  r = skipfile();
	    else
		  r = quick_skipfile();
	    
	    if (Writefile){
		  if (Physical)
			flushrec(DISKBLK,sout);
		  r += fclose(sout);
	    }
	    
	    r += readtail();		/* check trailer labels */
	    
	    (void) prtdir( &in, selectfile);

	    if (r)
		  return(r);
	    else if (!gotregex && namecnt==1)
		  break;	/* have all files */
      } 

      return(0);
}

int
query_dup_tape(void)
{
      char tmp[BUFSIZ];
      char *p;
      
      /* if Duplicate mode give option to append to existing tape */
      fprintf(stderr,"Copy files to end of existing tape (Y/N)? ");
      if (gets(tmp) == NULL)
	    return(-1);

      /* find first none blank character or end of line */
      for (p = tmp; isspace((int)*p) && *p != '\0'; p++)
	    ;
      
      if (*p == 'y' || *p == 'Y')
	    return(skip_tape(out.fd));
      else
	    return(newvol());
}

int
dup_tape(void)
{
      int nin, nout, r, bn;

      for (nin=nout=0; nin >=0 && nout >= 0; ) {
	    r = readhead();		/* read header labels */
	    if (r) return(r == -1 ? 0 : r);

	    /* copy selected files only */
	    Writefile = (*namelist==0) || wanted(in.tapefile);
	    
	    if (Writefile) {
		  /* write header using input tape data */
		  out.lblcnt = in.lblcnt;
		  (void) memcpy(&out.label1, &in.label1, 4*LABELSIZE);
		  if (r = writehead())
			return(r);

		  /* copy data blocks */
		  for (bn=1; (nin = read(in.fd,iobuf,in.block_len))>0; bn++) {
			fix2gblimit(in.fd);
			
			nout = write(out.fd,iobuf,nin);
			fix2gblimit(out.fd);
			
			if (nout != nin) {
			      fprintf(stderr,"ansitp read %d bytes but only wrote %d bytes\n",
				      nin,nout);
			      break;
			}
		  }
		  in.block_cnt = in.recd_cnt = --bn;
		  
		  if (nin < 0 || nout < 0) 
			perror("ansitp");

		  if (tapeop(out.fd, MTWEOF, 1))		/* EOF marker written */
			return(-1);
#ifdef SVR4
		  if (tapeop(in.fd, MTFSF, 1))                  /* skip read tape EOF mark */
			return(-1);
#endif
		  /* write tail records using input tape data */
		  if (r = readtail())
			return(r);
		  
		  out.lblcnt = in.lblcnt;
		  (void) memcpy(&out.label1, &in.label1, 4*LABELSIZE);
		  if (r = writetail())
			return(r);
	    }
	    else {
		  r = quick_skipfile();
		  if (r = readtail())		/* check trailer labels */
			return(r);
	    }
	    
	    (void) prtdir( &in, 1);
	    
	    if (!gotregex && namecnt==1)
		  break;	/* have all files */
      } 

      if (tapeop(out.fd, MTWEOF, 1))		/* EOT marker written */
	    return(-1);
      
      return(0);
}

/************************************************/
/*   locate and verify a header label on tape   */
/************************************************/
int
label_error(char *string)      /* print out message for error condition reading tape labels */
{
      fprintf(stderr,"ansitp: %s\n",string);
      fprintf(stderr,"ansitp: label image is -\n%80s\n",labelbuf);
      return(closetp(1));
}

void
strcat_lcase(char *dest, char *source)	/* lower case strcat func - used by readhead */
{
      /* position at end of destination string */
      while(*dest++)
	    ;
      --dest;
      /* copy source to destination */
      while(*dest++ = isupper(*source) ? tolower(*source++) : *source++);
      return;
}

int
readhead(void)	/* read and process header labels on tape */
{
      int n;		/* number of bytes returned on a read */
      int r;		/* general purpose returned value */
      int goodlbl;	/* count of correct labels read */
      char vfcflag;	/* len of fixed length prefix on variable len recds */
      
      in.lblcnt = Lineno = goodlbl = r = 0;
	    
      while(n = read(in.fd,labelbuf,LABELSIZE)) {
	    if (n < 0) {
		  perror("ansitp: read hdr");
		  return(closetp(1));
	    }
	    else if (n != LABELSIZE) {
		  fprintf(stderr,"ansitp: block length is %d\n",n);
		  return( label_error("header label block wrong length!") );
	    }
	    in.lblcnt++;	/* count labels */
	    
	    if (in.lblcnt == 1) {
		  if ((r=sscanf(labelbuf,"HDR1%17s%6s%4d%4d%*4d%*2d%2d%4d%2d%4d%*c%*6d",
				in.tapefile,in.filevol,&in.volseq,&in.filepos,&in.create_yy,
				&in.create_ddd,&in.expire_yy,&in.expire_ddd)) > 0) {
			if (r != 8) 
			      return( label_error("HDR1 label format error!") );
			goodlbl++;
			(void) memcpy(&in.label1,labelbuf,LABELSIZE);
			in.unixfile[0] = '\0';
			if (outdir != NULL) {
			      strcpy(in.unixfile,outdir);
			      strcat(in.unixfile,"/");
			}
			if (Ucase) strcat(in.unixfile,in.tapefile);
			else strcat_lcase(in.unixfile,in.tapefile);
			continue;
		  }
		  else {
			return( label_error("label sequence error - HDR1 label not first label") );
		  }			
	    }
	    else if (in.lblcnt == 2) {
		  if ((r=sscanf(labelbuf,"HDR2%c%5d%5d",&in.recd_fmt,
				&in.block_len,&in.recd_len)) > 0) {
			if (r != 3) 
			      return( label_error("HDR2 label format error!") );
			goodlbl++;
			(void) memcpy(&in.label2,labelbuf,LABELSIZE);
			continue;
		  }
		  else {
			return( label_error("label sequence error - HDR2 label not second label!") );
		  }
	    }
	    else if (in.lblcnt == 3) {
		  if ((r=sscanf(labelbuf,"HDR3%*25c%c",&vfcflag)) > 0) {
			if (r != 1) 
			      return( label_error("HDR3 label format error!") );
			goodlbl++;
			(void) memcpy(&in.label3,labelbuf,LABELSIZE);
			if (vfcflag == '2') Lineno = 1;	/* file has line nums */
			break;
		  }
		  else {
			return( label_error("label sequence error - HDR3 label not third label!") );
		  }
	    }
	    else if (in.lblcnt == 4) {
		  if (strncmp(labelbuf,"HDR4",4) == 0) {
			goodlbl++;
			(void) memcpy(&in.label4,labelbuf,LABELSIZE);
		  }
		  else {
			return( label_error("label sequence error - HDR4 label not fourth label!") );
		  }
	    }
	    
      }   /* end read header labels loop */
      
      if (n == 0 && in.lblcnt == 0)	/* two TM in a row - end of tape */
	    return(closetp(-1));	/* -1 means end of tape */

      /* print out complete information from file labels */
      if (Verbose > 1) 
	    prt_labels(&in);

      if (goodlbl != in.lblcnt) {
	    return( label_error("missing header label!") );
      }
      
      /* allocate a new I/O buffer if needed */
      if (iobuf == NULL || in.block_len > bufsize) {
	    if (iobuf != NULL) free(iobuf);
	    if ((iobuf = malloc((unsigned) in.block_len)) == NULL) {
		  fprintf(stderr,"ansitp: cannot allocate a %d char I/O buffer!\n",in.block_len);
		  return(closetp(1));
	    }
	    else
		  bufsize = in.block_len;
      }
      
      /* skip to start of data file */
#ifndef SVR4
      if (n == 0)
	    return(0);
#endif
      if (tapeop(in.fd, MTFSF, 1) != 0) {
	    fprintf(stderr,"ansitp: error skipping end of head tape filemark\n");
	    return(closetp(1));
      }
      
      return(0);
}

/***********************************************/
/*   set up to copy the current file on tape   */
/*   return 1 if readable 0 otherwise          */
/***********************************************/
int
setup(void)		/* set up to copy current file */
{
      
      if (in.recd_fmt != 'D' && in.recd_fmt != 'F' && in.recd_fmt != 'U') {
	    if (Writefile) {
		  fprintf(stderr,"ansitp: cannot convert record format '%c' files ",in.recd_fmt);
		  fprintf(stderr,"- format must be 'D', 'F', or 'U'\n");
	    }
	    Writefile = 0;
	    return(0);
      }
      
      if (in.recd_fmt == 'F' && (in.block_len % in.recd_len) != 0) {
	    if (Writefile) {
		  fprintf(stderr,"ansitp: block length (%d) must be a multiple of record length (%d)\n",
			  in.block_len,in.recd_len);
		  fprintf(stderr,"ansitp: for a record format 'F' file. File name is %s\n",
			  in.tapefile);
	    }
	    Writefile = 0;
	    return(0);
      }
      
      /* create output file */
      if (Writefile) {
	    if ((sout = fopen(in.unixfile,"w")) == NULL) {
		  fprintf(stderr,"ansitp: cannot create %s : ",in.unixfile);
		  perror("");
		  Writefile = 0;
	    } 
      }
      
      return(1);
} /* end setup */

/**********************/
/*   copy data file   */
/**********************/
int
reclen_D_fmt(char *p)
{
      char numchar[4];	/* string for atoi function */
      register int j;

      for (j=0; j<4; j++) {
	    numchar[j] = *p++;
	    if (! (isdigit(numchar[j]))) {
		  if (numchar[j] == '^')
			return(0); 
		  fprintf(stderr,"ansitp: non-numeric character in record header - %c\n",numchar[j]);
		  return(-2);	/* skip rest */
	    }
      }
      return( atoi(numchar) );  
}

int
copyfile(void)	/* copy current tape file to output */
{
      int bn;		/* block number */
      int j;		/* gen purpose counter */
      char *p;	        /* current position in iobuf */
      int rs;		/* record start index in iobuf */
      int rl;		/* record length */
      int dout;	        /* file descriptor for output file */
      int n = 0;	/* number of bytes returned on a read */
      int rn = 0;       /* record number */

      /* get unix file number associated with out file */      
      if(Writefile)
	    dout = fileno(sout);

      if (in.recd_fmt == 'F') {          /* FIXED block length format tape */
	    rl = in.recd_len;
	    for (bn=1; (n = read(in.fd,iobuf,bufsize))>0; bn++) {
		  fix2gblimit(in.fd);
		  for (p=iobuf,rs=0; rs<n; p+=rl,rs+=rl,rn++) {
			if (Writefile) {
			      j = rl;
			      if (Strip) {
				    while(p[j-1] == ' ')
					  j--;
				    if (j<0) j = 0;
			      }
			      if ((j) && (j != write(dout,p,j))) {
				    perror("ansitp: writing data file");
				    return(closetp(1));
			      }
			      if (Fnl) write(dout,"\n",1);
			}
		  }
	    }
	    in.block_cnt = --bn;
	    in.recd_cnt = rn;
      }
      else if (in.recd_fmt == 'U') {      /* U format tape */
	    for (bn=1; (n = read(in.fd,iobuf,bufsize))>0; bn++) {
		  fix2gblimit(in.fd);
		  if (Writefile)
			if (n != write(dout,iobuf,n)) {
			      perror("ansitp: writing data file");
			      return(closetp(1));
			}
	    }
	    in.block_cnt = in.recd_cnt = --bn;
      }
      else if (in.recd_fmt == 'D') {       /* D format tape: Variable record lengths */
	    /* read a block off tape into iobuf */
	    for (bn=1;(n = read(in.fd,iobuf,bufsize))>0; bn++) {
		  fix2gblimit(in.fd);
		  /* traverse thru iobuf */
		  for (p=iobuf,rs=0; rs<n; rs+=rl) {
			/* get record length out of record header */
			if ( (rl = reclen_D_fmt(p)) <= 0) {
			      if (rl == 0) break;  /* continue block-reading loop */
			      fprintf(stderr,"ansitp: block pos %d, block number %d\n",rs,bn);
 			      return(skipfile());
			}
			p += 4;                 /* advance p beyond rec header */
			rn++;			/* count the records */
			if (Writefile) {
			      if (Physical) {   /* copy physical record out */
				    fwriterec(p,DISKBLK,rl-4,sout);    
				    p += rl-4;
			      }
			      else {
				    j = 4;
				    if (Lineno) {
					  if (Addlnum) {
						/* add line numbers */
						j = *((short *) p);
						fprintf(sout,"%-8d",j);
					  }
					  p = p+2;
					  j = 6;
				    }
				    /* copy record out */
				    for (; j<rl; putc(*p,sout),p++,j++);
				    if (!No_nl) putc('\n',sout);
			      }
			}
			else
			      p += rl-4;
		  }
	    }
	    in.block_cnt = --bn;
	    in.recd_cnt = rn;
      }
      else {
	    in.block_cnt = in.recd_cnt = 0;
	    return(0);
      }

/* check exit status of last read */      
      if (n < 0) {
	    perror("ansitp: reading data file");
	    return(closetp(1));
      }
      
#ifdef SVR4
	if (tapeop(in.fd, MTFSF, 1) != 0) {
	      fprintf(stderr,"ansitp: error skipping eod tape filemark\n");
	      return(closetp(1));
	}
#endif
      return(0);

}	/* end copyfile */

/*************************************************************/
/*   skip data file  ... by reading one tape block at a time */
/*************************************************************/
int
skipfile(void)	/* skip unwanted data file */
{
	int n;		/* number of bytes returned on a read */

	in.recd_cnt = in.block_cnt = 0;

	while(n = read(in.fd,iobuf,bufsize)) {
		if (n < 0) {
			perror("ansitp: skip unwanted data file");
			return(closetp(1));
		}
		in.block_cnt++;
	}
	
#ifdef SVR4
	if (tapeop(in.fd, MTFSF, 1) != 0) {
	      fprintf(stderr,"ansitp: error skipping eod tape filemark\n");
	      return(closetp(1));
	}
#endif
	return(0);
}	/* end skipfile */

int
quick_skipfile(void)	/* skip unwanted data file */
{
      
      in.recd_cnt = in.block_cnt = 0;
      
      if (tapeop(in.fd, MTFSF, 1) != 0) {
	    fprintf(stderr,"ansitp: error skipping eod tape filemark\n");
	    return(closetp(1));
      } 
      return(0);
}	/* end quick_skipfile */

/******************************************************/
/*   verify a correct trailer label after data file   */
/******************************************************/
int
readtail(void)	/* read and process trailer labels on tape */
{
	int n;		/* number of bytes returned on a read */
	int r;		/* general purpose returned value */
	char tempname[18];	/* filename from trailer label */
	char eov;	/* type of EOF1 label */
	int cnt;	/* block count from trailer label */
	int goodlbl;	/* count of correct labels read */

	in.lblcnt = goodlbl = r = 0;

	while(n = read(in.fd,labelbuf,LABELSIZE)) {
	      if (n < 0) {
		    perror("ansitp: read trailer");
		    return(closetp(1));
	      }
	      if (n != LABELSIZE) {
		    fprintf(stderr,"ansitp: block length is %d\n",n);
		    return( label_error("trailer label block wrong length!") );
	      }
	      in.lblcnt++;	/* count labels */
	      
	      if (in.lblcnt == 1) {
		    if ((r=sscanf(labelbuf,"EO%c1%17s%*6s%*4d%*4d%*4d%*2d%*6d%*6d%*c%6d",
				  &eov,tempname,&cnt)) > 0) {
			  if (r != 3 || (eov != 'F' && eov != 'V')) 
				return( label_error("EOF1 label format error!") );
			  goodlbl++;
			  (void) memcpy(&in.label1,labelbuf,LABELSIZE);
			  if (strcmp(in.tapefile,tempname) != 0) {
				fprintf(stderr,"ansitp: filename on trailer label (%s)\n",tempname);
				fprintf(stderr,"ansitp: does not match header label! (%s)\n",in.tapefile);
				return(closetp(1));
			  }
			  if (Scantape) 
				in.block_cnt = cnt;
		    }
		    else {
			  return( label_error("label sequence error - EOF1 label not first label!") );
		    }
	      }
	      else if (in.lblcnt == 2) {
		    if ((r=sscanf(labelbuf,"EO%c2", &eov)) > 0) {
			  if (eov != 'F' && eov != 'V') 
				return( label_error("EOF2 label format error!") );
			  goodlbl++;
			  (void) memcpy(&in.label2,labelbuf,LABELSIZE);
		    }
		    else {
			 return( label_error("label sequence error - EOF2 label not second label!") );
		    } 
	      }
	      else if (in.lblcnt == 3) {
		    if ((r=sscanf(labelbuf,"EO%c3", &eov)) > 0) {
			  if (eov != 'F' && eov != 'V') 
				return( label_error("EOF3 label format error!") );
			  goodlbl++;
			  (void) memcpy(&in.label3,labelbuf,LABELSIZE);
		    }
		    else {
			 return( label_error("label sequence error - EOF3 label not third label!") );
		    }
	      }
	      else if (in.lblcnt == 4) {
		    if ((r=sscanf(labelbuf,"EO%c4", &eov)) > 0) {
			  if (eov != 'F' && eov != 'V') 
				return( label_error("EOF4 label format error!") );
			  goodlbl++;
			  (void) memcpy(&in.label4,labelbuf,LABELSIZE);
		    }
		    else {
			 return( label_error("label sequence error - EOF4 label not fourth label!") );
		    }
	      }
	}   /* end read tail labels loop */

	if (in.lblcnt != goodlbl) 
	      return( label_error("missing trailer label!") );
	
	/* skip to next header */
#ifndef SVR4
	if (n == 0)
	      return(0);
#endif
	if (tapeop(in.fd, MTFSF, 1) != 0) {
	      fprintf(stderr,"ansitp: error skipping end of trail tape filemark\n");
	      return(closetp(1));
	}	

	return(0);
}

/*********************************************/
/*   close tape file and check return code   */
/*********************************************/
int
closetp(int rc)
{
      if (in.open == TRUE) {
	    (void) close(in.fd);
	    in.open = FALSE;
      }
      if (out.open == TRUE) {
	    (void) close(out.fd);
	    out.open = FALSE;
      }
      return(rc);
}


/**************************************************/
/*   copy regular expressions and reformat them   */
/**************************************************/
int
copy_regex(char *arglist[])
{  
	char **a,**n,*p,*s, *workspace;
	FLAG *f;
	int charcnt = 0;
	int maxlen = 0;

	/* ensure global variables to zero */
	gotregex=namecnt= 0;

	/* count up space needed to copy the expressions */
	/* and determine if we have any regular expressions */
	for  (a=arglist; *a ; a++) {
		namecnt++;
		for (p=(*a); *p; p++) {
			charcnt++;
			switch(*p) {
			case '.':
				charcnt++;  /* will need another char */
				if (!gotregex) break;
			case '*':
				charcnt++;  /* will need another char */
			case '\\':
			case '[':
			case '?':
			case '^':
			case '$':
				gotregex++;
				break;
			}
		}
		maxlen = (charcnt > maxlen) ? charcnt : maxlen;
	}
	if (namecnt == 0) {
		namelist = arglist;
		return(0);
	}

	namecnt++;	/* allow for null ptr at end */

	/* allocate memory for name lists */
	nameflag = (FLAG *)  calloc((unsigned) namecnt,sizeof(*nameflag));
	if (nameflag == NULL) {
	    fprintf(stderr, "ansitp: can't alloc space for %d nameflags\n",
		namecnt);
	    return(-1);
	}
	namelist = (char **) calloc((unsigned) namecnt,sizeof(*namelist));
	if (namelist == NULL) {
	    fprintf(stderr, "ansitp: can't alloc space for %d names\n",
		namecnt);
	    return(-1);
	}
	workspace = (char *) alloca((size_t) maxlen);
	if (workspace == NULL) {
	    fprintf(stderr, "ansitp: can't alloca space for %d chars workspace\n",
		charcnt);
	    return(-1);
	}

	a = arglist;	/* ptr to an array of string pointers (*argv[]) */
	n = namelist;	/* ptr to an array of string pointers (output) */
	f = nameflag;	/* ptr to an array of flags */

	/* convert all reg expr to format suitable for regex */
	/* striaght forward copy for names without reg expr */
	/* ? --> .  */
	/* . --> \. */
	/* * --> .* */
	for  ( ; *a; a++,n++,f++) {
	      s = workspace;
	      *f = 0;		/* set flag to assume not a reg expression */
	      /* reformat reg expression for regex pgms */
	      for (p=(*a); *p; p++,s++) {
		    if (!gotregex) {
			  *s = (Ecase || !islower(*p)) ? *p : toupper(*p);
		    }
		    else {
			  switch(*p) {
			     case '?':
				*s = '.';
				*f = 1;		/* is a reg expr */
				break;
			     case '.':
				*s++ = '\\';
				*s = '.';
				*f = 1;		/* is a reg expr */
				break;
			     case '*':
				*s++ = '.';
				*s = '*';
				*f = 1;		/* is a reg expr */
				break;
			     case '\\':
			     case '[':
			     case '^':
			     case '$':
				*s = *p;
				*f = 1;		/* is a reg expr */
				break;
			     default:
				*s = (Ecase || !islower(*p)) ? *p : toupper(*p);
			  }
		    }
	      }
	      *s++ = '\0';
#ifdef SUNOS
	      if ((p = re_comp(*n)) != NULL) {
		    /* error in regular expression */
		    fprintf(stderr,"ansitp: regexpr error in \"%s\" : %s\n",*n,p);
		    return(1);
	      }
	      if ( (*n = strdup(workspace)) == NULL) {
		    fprintf(stderr,"ansitp: can't alloc space for name\n");
		    return(-1);
	      }
#else
	      if (*f) {     /* if expression is reg expr store compiled version */
		    if ((*n = compile(workspace, NULL, NULL)) == NULL) {
			  /* error in regular expression */
			  fprintf(stderr,"ansitp: regexpr error in \"%s\"",workspace);
			  return(1);
		    }
	      }
	      else {
		    if ( (*n = strdup(workspace)) == NULL) {
			  fprintf(stderr,"ansitp: can't alloc space for name\n");
			  return(-1);
		    }
	      }
#endif
	}
	*n = 0;			/* flag end of list */
	return(0);
}

/******************************************************/
/*   check namelist to see if user wants this file    */
/******************************************************/
int
wanted(char *file)
{  
      char **n;
      FLAG *f;
      
      /* scan namelist to see if we want this file */
      for  (n=namelist,f=nameflag; *n ; n++,f++) {
	    if (gotregex && *f==1) {
#ifdef SUNOS
		  re_comp(*n);		/* compile reg expr */
		  if (re_exec(file))    /* test file against current compiled reg expr */
			return(1);
#else		      
		  if (step(file, *n))
			return(1);
#endif			  
	    }
	    else if (*f==0 && strcmp(*n,file)==0) {
		  namecnt--;		/* one less file */
		  *f |= 2;		/* skip this entry next time */
		  return(1);
	    }
      }
      return(0);
}

/******************************/
/*   print a directory line   */
/******************************/
int
prtdir(ansitp_info_t *cur, FLAG flag)	/* print directory line */
{
      static FLAG titled = 0;
      char sosfmt,*unixname;
      
      sosfmt = Lineno ? '*' : ' ';
      unixname = (Writefile) ? cur->unixfile : "";

      if (!Verbose && !Dirflag && !flag )
	    return(0);

      if (Scantape) {
	    printf("%-20sFmt=%c%c Block len %-5d Fileset %-6.6s File pos %-4d Blocks %-6d\n",
		   cur->tapefile,cur->recd_fmt,sosfmt,cur->block_len,
		   cur->label1.fileset,cur->filepos,cur->block_cnt);
	    return(0);
      }
	  
      if ( (flag | Dirflag) == 1)  {
	    if (!titled) {
		  titled++;
		  printf("Tape file name        Blocks  Records  Format   Unix filename\n");
	    }
	    printf("%-20s%8d %8d %5c%c    %s\n",
		   cur->tapefile,cur->block_cnt,cur->recd_cnt,cur->recd_fmt,sosfmt,unixname);
      }
      else {
	    printf("\n\n%-20sFile pos %-4d  Records %-6d  Blocks %-6d  Created %d.%d\n",
		   cur->tapefile,cur->filepos,cur->recd_cnt,cur->block_cnt,
		   cur->create_yy,cur->create_ddd);
	    printf("%20sFormat %c%c   Block len %-5d  Record len %-5d  Expires %d.%d\n","",
		   cur->recd_fmt,sosfmt,cur->block_len,cur->recd_len,
		   cur->expire_yy,cur->expire_ddd);
	    
	    if (*unixname != '\0') printf("copied to %s\n",unixname);
      }
      return(0);
}

/*********************************************/
/*   print out ANSI labels in verbose mode   */
/*********************************************/
void
prt_labels(ansitp_info_t *cur)
{
      int nos = (cur->lblcnt <= 4) ? cur->lblcnt : 4;
      if (nos-- > 0) {
	    ansi_hdr1_t *l = &cur->label1;
	    fprintf(stderr,"\n\nANSI label1:\n\tfile name = %-.17s fileset = %-.6s\n",
		    l->name,l->fileset);
	    fprintf(stderr,"\tvol nos = %-.4s file nos = %-.4s generation = %-.4s %-.2s\n",
		    l->volume_num,l->file_num,l->gen,l->genver);
	    fprintf(stderr,"\tcreated %-.6s on system %-.13s expires %-.6s access = '%c'\n",
		    l->created,l->tapesys,l->expires,l->access);
      }
      if (nos-- > 0) {
	    ansi_hdr2_t *l = &cur->label2;
	    fprintf(stderr,"ANSI label2:\n\tblocklen = %-.5s reclen = %-.5s format = %c\n",
		    l->blocklen,l->reclen,l->recfmt);
	    fprintf(stderr,"\toffset = %-.2s density = '%c' blocking = '%c' vol switch = '%c'\n",
		    l->block_offset,l->density,l->blocked_records,l->vol_switch);
      }
      if (nos-- > 0) {
	    fprintf(stderr,"ANSI label3:\n\t%-.80s\n",(char *) &cur->label3);
      }
      if (nos-- > 0) {
	    fprintf(stderr,"ANSI label4:\n\t%-.80s\n",(char *) &cur->label4);
      }
      return;
}

/********************************************************/
/*	Newvol() - write a volume label to the tape	*/
/********************************************************/
int
newvol(void)
{
      register char *cp;
      char tmp[100];
      ansi_vol1_t *vol1 = (ansi_vol1_t *) labelbuf;

      /* get new volume label */
      fprintf(stderr,"Volume label? ");
      if (gets(tmp) == NULL)
	    return(-1);
      
      if (tmp[0]!='\0') {		
	    cp = tmp;
	    /* convert to upper case */
	    while(*cp) {		
		  if (islower(*cp))
			*cp = toupper(*cp);
		  cp++;
	    }
	    
	    strncpy(out.volname, tmp, 6);
	    out.volname[6] = '\0';
      }
      
      else			/* no name, use "UNIX" */
	    strcpy(out.volname, "UNIX");
      
      if ( (cp = getenv("LOGNAME")) == NULL)
	    cp = "NONAME";
      
      (void) memset(labelbuf, ' ', LABELSIZE);
      (void) sprintf(labelbuf,"VOL1%-6s",out.volname);
      (void) sprintf(vol1->owner,"%-14.14s",cp);
      vol1->access = ' ';
      vol1->ansi_level = '3';
		  
      if (write(out.fd, labelbuf, LABELSIZE) != LABELSIZE) {
	    perror("ansitp: error writing VOL1");
	    return(-1);
      }
      out.filepos = 0;
      return(0);
}


/********************************************************/
/*	skiptape() - skip all the files on the tape	*/
/********************************************************/
int
skip_tape(int fd)
{
    struct mtget status; 
    if (tapeop(fd, MTEOM, 0) != 0)
	  return(-1);	/* move to end of medium */
    if (tapeop(fd, MTBSF, 1) != 0)
	  return(-1);	/* back up over 1 FILEMARK at EOM */
    if (ioctl(fd, MTIOCGET, (char *)&status)) {
	  perror("ansitp: tape status in skip_tape");
	  return(-1);
    }
    out.filepos = status.mt_fileno/3;
    return(0);
    }


/****************************************************/
/* tapeop(fd, cmd, count) - execute tape ioctl      */
/****************************************************/
int
tapeop(int fd, int cmd, int count)
{
      struct mtop mopp;
      
      mopp.mt_op = cmd;
      mopp.mt_count = count;
      if (ioctl(fd, MTIOCTOP, &mopp)) {
	    perror("ansitp: tape control");
	    return(-1);
      }
      return(0);
}     

/************************************************/
/* writape() - write the files onto the tape	*/
/************************************************/
int
write_tape(void)
{
      long ticks;
      char tmp[LABELSIZE + 20];
      struct tm *tmtime;
      int rc;
      
      if (Fixed) {
	    printf("Fixed-length record size? ");
	    if (gets(tmp) == NULL) return(-1);
	    out.recd_len = atoi(tmp);
	    if (out.recd_len <= 0 || out.recd_len > 16384) {
		  if (Padd) out.recd_len = 80;
		  else out.recd_len = 512;
	    }
	    
	    out.block_len = (TAPEBLK / out.recd_len) * out.recd_len;
	    if (TAPEBLK % out.block_len != 0)
		  out.block_len += out.recd_len;
	    
	    out.recd_fmt = 'F';
      }
      else {			/* variable length format */
	    out.recd_len = TAPEBLK;
	    out.block_len = TAPEBLK;
	    out.recd_fmt = 'D';
      }
      
      time(&ticks);		/* get the year and day-of-year */
      tmtime = localtime(&ticks);
      out.create_yy= tmtime->tm_year;
      out.create_ddd = tmtime->tm_yday;
      
      if (iobuf == NULL || out.block_len > bufsize) {
	    if (iobuf != NULL)		/* get a buffer for output */
		  free(iobuf);
	    iobuf = malloc((unsigned) out.block_len);
	    bufsize = out.block_len;
      }
      if (iobuf == NULL) {
	    fprintf(stderr,"ansitp: cannot allocate a %d char Output buffer!\n",
		    out.block_len);
	    return(-1);
      }
      
      if (!Fixed) {
	    if (Ibuf != NULL)		/* get a buffer for input */
		  free(Ibuf);
	    Ibuf = malloc((unsigned) out.recd_len);
	    if (Ibuf == NULL) {
		  fprintf(stderr,"ansitp: cannot allocate a %d char Input buffer!\n",
			  out.recd_len);	
		  return(-1);
	    }
      }
      
      Writefile++;
      out.filepos++;
      while(namecnt--) {		/* write the files, one by one */
	    if (*namelist == NULL || **namelist == '\0')
		  continue;
	    if ((sout = fopen(*namelist, "r")) == NULL) {
		  fprintf(stderr, "ansitp: Can't open %s for reading!\n", *namelist);
		  perror("");
		  namelist++;
		  continue;
	    }
	    
	    bestname(*namelist, out.tapefile);	/* get a good tape label name */
	    
	    if (writehead()) {
		  fclose(sout);
		  return(-1);
	    }

	    if (Fixed)
		  rc = fixout();	/* fixed length out */
	    else
		  rc = varout();		/* variable length */
	    if (rc)
		  return(-1);
	    
	    if (tapeop(out.fd, MTWEOF, 1))		/* EOF marker written */
		  return(-1);
	    
	    if (writetail()) 
		  return(-1);

	    if (Verbose) {			/* print out name on tape */
		  strncpy(out.unixfile, *namelist, 49);
		  out.unixfile[49] = '\0';
		  prtdir(&out, 1);
	    }
	    
	    namelist++;		/* get next file name */
	    out.filepos++;
      }

      tapeop(out.fd, MTWEOF, 1);		/* EOT marker */
      return(0);
}

int
writehead(void)
{
      if (Duplicate) {
	    int i;
	    ansi_hdr1_t *lptr = &out.label1;
	    for (i=0; i < out.lblcnt && i < 4; i++) {
		  if (write(out.fd, lptr, LABELSIZE) != LABELSIZE) {
			perror("ansitp: error writing HDR1");
			return(-1);
		  }
		  lptr++;
	    }
      }
      else {       /* use default of two labels */
	    char tmp[LABELSIZE + 20];
	    sprintf(tmp,
		    "HDR1%-17s%-6s0001%04d000100%3d%03d 00000 000000DECFILE11A          ",
		    out.tapefile, out.volname, out.filepos,
		    out.create_yy, out.create_ddd);
	    
	    if (write(out.fd, tmp, LABELSIZE) != LABELSIZE) {
		  perror("ansitp: error writing HDR1");
		  return(-1);
	    }
	    sprintf(tmp,
		    "HDR2%c%05d%05d                                   00                            ",
		    out.recd_fmt, out.block_len, out.recd_len);
	    
	    if (write(out.fd, tmp, LABELSIZE) != LABELSIZE) {
		  perror("ansitp: error writing HDR2");
		  return(-1);
	    }
      }
      
      /* write EOF marker after labels */
      if (tapeop(out.fd, MTWEOF, 1)) {		
	    return(-1);
      }
      
      return(0);
}

int
writetail(void)
{
      if (Duplicate) {
	    int i;
	    ansi_hdr1_t *lptr = &out.label1;
	    for (i=0; i < out.lblcnt && i < 4; i++) {
		  if (write(out.fd, lptr, LABELSIZE) != LABELSIZE) {
			perror("ansitp: error writing HDR1");
			return(-1);
		  }
		  lptr++;
	    }
      }
      else {       /* use default of two labels */
	    char tmp[LABELSIZE + 20];
            sprintf(tmp,
		    "EOF1%-17s%-6s0001%04d000100%3d%03d 00000 %06dDECFILE11A          ",
		    out.tapefile, out.volname, out.filepos, out.create_yy,
		    out.create_ddd, out.block_cnt);
	    
	    if (write(out.fd, tmp, LABELSIZE) != LABELSIZE) {
		  perror("ansitp: error writing EOF1");
		  return(-1);
	    }
	    sprintf(tmp,
		    "EOF2%c%05d%05d                                   00                            ",
		    out.recd_fmt, out.block_len, out.recd_len);
	    
	    if (write(out.fd, tmp, LABELSIZE) != LABELSIZE) {
		  perror("ansitp: error writing EOF2");
		  return(-1);
	    }
      }

      /* write EOF marker after labels */
      if (tapeop(out.fd, MTWEOF, 1))		
	    return(-1);
      
      return(0);
}
/********************************************************/
/* bestname(src, dst) - copies src to dst, but removes	*/
/* all directory prefixes first.  It also translates	*/
/* to upper case and uses the first 10 and last 7 chars	*/
/* if the result is more than 17 characters.		*/
/********************************************************/
int
bestname(char *src, register char *dst)
{
      register char *cp;
      int i;
      
      cp = strrchr(src, (int) '/');	 /* find last directory prefix */
      if (cp == NULL) cp = src;
      else cp++;
      
      for (i=0; i<10 && *cp; i++, cp++) {	/* copy first 10 chars as is */
	    if (!Ucase && islower(*cp))
		  *dst++ = toupper(*cp);
	    else *dst++ = *cp;
      }
      
      if (*cp == '\0') {			/* less than 10 chars, so leave */
	    *dst = *cp;
	    return(0);
      }
      
      if ((i = strlen(cp)) > 17)		/* if more than 17 chars, take last 7 */
	    cp += (i - 7);
      
      while(*cp) {
	    if (!Ucase && islower(*cp))
		  *dst++ = toupper(*cp);
	    else *dst++ = *cp;
	    cp++;
      }
      *dst = *cp;
      return(0);
}


/********************************************************/
/* varout() - write the file in variable length format	*/
/********************************************************/
int
varout(void)
{
      register char *cp;
      register int ccount,i;
      
      cp = iobuf;
      ccount = 0;
      out.recd_cnt = 0;
      out.block_cnt = 0;
      
      while ((i=((Physical)
		 ?freadrec(Ibuf,DISKBLK,out.recd_len,sout)
		 :fgetline(Ibuf, out.recd_len, sout)))!= 0)
	    {	/* read a record */
		  out.recd_cnt++;
		  
		  if (!Physical){
			if (Ibuf[i-1] == '\n') {		/* remove newline */
			      i--;
			      Ibuf[i] = '\0';
			}
		  }
		  i += 4;		/* add 4 for character count */
		  
		  if ((ccount+i) > out.block_len) {	
			/* too big to fit, write block out */
			out.block_cnt++;
			
			while (ccount++ < out.block_len)	/* padd block out */
			      *cp++ = '^';
			
			if (write(out.fd, iobuf, out.block_len) != out.block_len) {
			      perror("ansitp: error writing tape");
			      fclose(sout);
			      return(-1);
			}
			fix2gblimit(out.fd);
			
			cp = iobuf;
			ccount = 0;
		  }
		  
		  sprintf(cp, "%04d", i);
		  cp += 4;
		  (Physical) ? memcpy(cp, Ibuf, i-4) : strcpy(cp, Ibuf);
		  cp += (i - 4);
		  ccount += i;
	    }
      
      if (cp != iobuf) {		/* write the remainder */
	    out.block_cnt++;
	    
	    while (ccount++ < out.block_len)	/* padd block out */
		  *cp++ = '^';
	    
	    if (write(out.fd, iobuf, out.block_len) != out.block_len) {
		  perror("ansitp: error writing tape");
		  fclose(sout);
		  return(-1);
	    }
      }
      
      fclose(sout);
      return(0);
}


/****************************************/
/* fixout - write fixed length records	*/
/****************************************/
int
fixout(void)
{
      register char *cp;
      register int ccount,i;
      
      cp = iobuf;
      ccount = 0;
      out.recd_cnt = 0;
      out.block_cnt = 0;
      
      if (Padd) {
	    if (fgets(cp, out.recd_len, sout) == NULL) i = 0;
	    else i = strlen(cp);
      }
      else
	    i = fread(cp, 1, out.recd_len, sout);
      
      while(i) {
	    out.recd_cnt++;
	    
	    if (Padd) {
		  if (cp[i-1] == '\n') {      /* remove newline */
			i--;
		  }
	    }
	    cp += i;
	    while(i < out.recd_len) {
		  *cp++ = ' ';		      /* padd record with blanks */
		  i++;
	    }
	    
	    ccount += i;
	    
	    if (ccount >= out.block_len) {
		  /* too big to fit, write block out */
		  out.block_cnt++;
		  
		  if (write(out.fd, iobuf, out.block_len) != out.block_len) {
			perror("ansitp: error writing tape");
			fclose(sout);
			return(-1);
		  }
		  fix2gblimit(out.fd);
		  
		  cp = iobuf;
		  ccount = 0;
	    }
	    
	    if (Padd) {
		  if (fgets(cp, out.recd_len, sout) == NULL) i = 0;
		  else i = strlen(cp);
	    }
	    else
		  i = fread(cp, 1, out.recd_len, sout);
      }
      
      if (cp != iobuf) {		/* write the remainder */
	    out.block_cnt++;
	    
	    while (ccount++ < out.block_len)	/* padd block out */
		  *cp++ = '\0';
	    
	    if (write(out.fd, iobuf, out.block_len) != out.block_len) {
		  perror("ansitp: error writing tape");
		  fclose(sout);
		  return(-1);
	    }
      }
      
      fclose(sout);
      return(0);
}

int
fgetline(char *line,int max,FILE *fp)
{
      if (fgets(line,max,fp) == NULL)
	    return 0;
      else
	    return strlen(line);
}
      
static void
fix2gblimit(int fd )
{
/* hack to get round UNIX 2Gb file limit :			*/
/* just reset file pointer to zero when it goes over 1Gbyte */
	if( lseek( fd, 0L, SEEK_CUR ) > 1073741824 )
		(void) lseek( fd, 0L, SEEK_SET );
}
