#include <stdio.h>
#include <math.h>
#include <string.h>
#include <memory.h>
#include <signal.h>

#define MAIN
#include "globals.h"

#define CALC_EVAL    1
#define CALC_LOAD    2
#define CALC_HISTO   3
#define CALC_FOURIER 4
#define CALC_RUN     5
#define CALC_REG     6
#define CALC_DIFF    7
#define CALC_XCOR    8
#define CALC_SPLINE  9
#define CALC_SAMP    10
#define CALC_DIGF    11
#define CALC_LCONV   12
#define CALC_LEVAL   13
#define CALC_NONL    14

extern void  cxfree();
extern void  output_data();
extern void  errwin();
extern void  sig_fault();
extern void  end_proc();
static void  forwarddiff();
static void  backwarddiff();
static void  centereddiff();

static char *reply, *in_file;
	
int
main(argc, argv)
      int argc;
      char *argv[];
{
      FILE   *fp_in, *fp_out;
      char   *command;
      int    i, ierr, option, subset1, subset2 = -1;
      int    set1, set2, setlen1, setlen2;
      char   start_str[BUFSIZ], stop_str[BUFSIZ];
      double start, stop;
      
      for (i=1; argv[i]; i++) {
	    if (strcmp(argv[i],"-spectrum") == 0)
		  in_file = argv[i+1];
      }
      if (in_file == NULL) {
      	    fprintf(stderr,"no input file specified\n");
	    return;
      }

/* set up signal handler for fault conditions ... lets not produce core files */      
      (void) signal(SIGSEGV,sig_fault);
      (void) signal(SIGBUS,sig_fault);
	    
/* allocate memory for command fields */
      if ( (command = (char *) calloc(BUFSIZ,sizeof(char))) == NULL) {
	    fprintf(stderr,"%s: couldn't allocate memory\n",argv[0]);
	    end_proc(-1);
      }
      if ( (reply = (char *) calloc(BUFSIZ,sizeof(char))) == NULL) {
	    fprintf(stderr,"%s: couldn't allocate memory\n",argv[0]);
	    end_proc(-1);
      }

/* read in spectrum data from data file */      
      if ( (fp_in = fopen(in_file,"r")) == NULL) {
	    fprintf(stderr,"%s: failed to open data file\n",argv[0]);
	    end_proc(-1);
      }
      if ( fread(command, BUFSIZ, 1, fp_in) != 1) {
	    fprintf(stderr,"%s: Read from data file failed\n",argv[0]);
	    end_proc(-1);
      }

/* scan first part of command string to get option */
      sscanf(command,"%d %d %d %s %s",&option, &setlen1, &setlen2, start_str, stop_str);
      start = atof(start_str);
      stop = atof(stop_str);

/* always need to get at least one set of data ... so get it here */      
      if ( (set1 = new_set(setlen1)) == -1) 
	    end_proc(-1);
      if ( fread((char *) getsetx(set1), sizeof(double), setlen1, fp_in) != setlen1) {
	    fprintf(stderr,"%s: Read from spline data file failed\n",argv[0]);
	    end_proc(-1);
      }
      if ( fread((char *) getsety(set1), sizeof(double), setlen1, fp_in) != setlen1) {
	    fprintf(stderr,"%s: Read from spline data file failed\n",argv[0]);
	    end_proc(-1);
      }
      nsort_xy(set1,0,0);
      if ( (subset1 = select_subset(set1,start,stop)) == -1)
	    end_proc(-1);
      
/* check to see whether there is another set to read */
      if (setlen2 > 0) {
	    if ( (set2 = new_set(setlen2)) == -1) 
		  end_proc(-1);
	    if ( fread((char *) getsetx(set2), sizeof(double), setlen2, fp_in) != setlen2) {
		  fprintf(stderr,"%s; Read from spline data file failed\n",argv[0]);
		  end_proc(-1);
	    }
	    if ( fread((char *) getsety(set2), sizeof(double), setlen2, fp_in) != setlen2) {
		  fprintf(stderr,"%s: Read from spline data file failed\n",argv[0]);
		  end_proc(-1);
	    }
	    nsort_xy(set2,0,0);
	    if ( (subset2 = select_subset(set2,start,stop)) == -1)
		  end_proc(-1);
      }	          

/*
 *       choose method to deal with option
 *       methods of form:
 *       calc(FILE *fp_in, char *c_ptr, int setno1, int setno2);
 */
      switch(option) {
	    case CALC_EVAL:
/*	    fprintf(stderr,"performing eval option\n");   */
	    ierr = calc_eval(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_LOAD:
	    ierr = -1;
	    break;
	    
	    case CALC_HISTO:
/*	    fprintf(stderr,"performing histo option\n");   */
	    ierr = calc_histo(fp_in,command,subset1,subset2);
	    break;

	    case CALC_FOURIER:
/*	    fprintf(stderr,"performing fourier option\n");   */
	    ierr = calc_fourier(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_RUN:
/*	    fprintf(stderr,"performing run option\n");   */
	    ierr = calc_runa(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_REG:
/*	    fprintf(stderr,"performing reg option\n");   */
	    ierr = calc_reg(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_DIFF:
/*	    fprintf(stderr,"performing diff option\n");   */
	    ierr = calc_diff(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_XCOR:
/*	    fprintf(stderr,"performing xcor option\n");     */
	    ierr = calc_xcor(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_SPLINE:
/*	    fprintf(stderr,"performing spline option\n");    */
	    ierr = calc_spline(fp_in,command,subset1,subset2); 
	    break;

	    case CALC_SAMP:
/*	    fprintf(stderr,"performing samp option\n");     */
	    ierr = calc_samp(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_DIGF:
/*	    fprintf(stderr,"performing digf option\n");    */
	    ierr = calc_digf(fp_in,command,subset1,subset2);
	    break;

	    case CALC_LCONV:
/*	    fprintf(stderr,"performing lconv option\n");    */
	    ierr = calc_lconv(fp_in,command,subset1,subset2);
	    break;
	    
	    case CALC_LEVAL:
	    ierr = -1;
	    break;

	    case CALC_NONL:
/*	    fprintf(stderr,"performing nonl option\n"); */
	    ierr =calc_nonl(fp_in,command,subset1,subset2);
	    break;

	    default:
	    fprintf(stderr,"%s: unrecognised command option %d\n",argv[0],option);
	    end_proc(-1);
      }
	    
      fprintf(stderr,"finished calculating\n");
      (void) fclose(fp_in);
      if (ierr == 0) {
	    output_data(reply,BUFSIZ,getsetx(subset1),getsety(subset1),getsetlen(subset1));
      }
      else
	    fprintf(stderr,"%s: error occurred in calculation\n",argv[0]);
      end_proc(ierr);
}

int
calc_eval(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      char tmpstr[512], *ptr = command;
      int i = 0, errpos, len;
      double *xtmp, *ytmp;
      char *ptr2;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      ptr2 = ptr;           /* keep for output command */
      ptr += strlen(ptr) + 1;
      strcpy(tmpstr,ptr);
      
      len = getsetlen(set1);
      xtmp = getsetx(set1);
      ytmp = getsety(set1);

      fixupstr(tmpstr);
      scanner(tmpstr, xtmp, ytmp, len, ax, bx, cx, dx, MAXARR, i, set1, &errpos); 

      sprintf(reply,"%d %c using formula %s %c %s",len,'\0',tmpstr,'\0',ptr2);
      return( errpos);
}

int
calc_histo(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      char *ptr = command;
      int *ind, hist_type, nbins;
      int tmpset, i, j, len, binmax, n1;
      double spread, binw, xmin, xmax, sum = 0.0;
      double *xdata, *ydata, *xtmp, *ytmp;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%lg %d %lg %lg",&binw,&hist_type, &xmin, &xmax);
      ptr += strlen(ptr) + 1;
      
      len = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);
      
/* determine binning parameters*/
      spread = xmax - xmin;
      nbins = (int) (spread / binw);
      if (nbins <= 0) {
	    errwin("Bin size <= 0, no work to do");
	    return(-1);
      }
      
/* get new set to store final results in */
      if ( (tmpset = new_set(nbins)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform calculation */  
      ind = (int *) calloc(len, sizeof(int));
      if (ind == NULL) {
	    errwin("Not enough memory for histogram");
	    return;
      }
      for (j = i = 0; i < len; i++) {
	    j = (int) ((ydata[i] - xmin) / binw);
	    ind[j] = ind[j] + 1;
      }
      n1 = 0;
      binmax = 0;
      for (i = 0; i < nbins; i++) {
	    xtmp[i] = i * binw + xmin;
	    sum = sum * hist_type + ind[i];	/* hist_type = 0 => regular histo */
	    ytmp[i] = sum;
	    n1 = n1 + ind[i];
	    if (ind[i] > binmax) {
		  binmax = ind[i];
	    }
      }

      sprintf(reply,"%d %c %s",nbins,'\0',ptr);
      cxfree(ind);
      return(swap_sets(set1,tmpset));
}

int
calc_fourier(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int i, len, tmpset, tmplen;
      int imode, load, loadx, invflag, type, wind;
      char *ptr = command;
      double *x, *y, *xx, *yy, delt, T;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d %d %d %d %d %d", &imode, &load, &loadx, &invflag, &type, &wind);
      ptr += strlen(ptr) + 1;

/* get data set parameters */
      tmplen = len = getsetlen(set1);
      x = getsetx(set1);
      y = getsety(set1);
     
/* get new set to store final results in */
      if ( (tmpset = new_set(tmplen)) == -1)
	    return(-1);
      xx = getsetx(tmpset);
      yy = getsety(tmpset);
      if (copyset(set1,tmpset) == -1)
	    return(-1);
      
/* perform calculation */
      if (wind != 0) {	/* apply data window if needed */
	    apply_window(xx, yy, len, type, wind);
      }

      if (imode <= 1) {
	    if (type == 0) {	/* real data */
		  for (i = 0; i < len; i++) {
			xx[i] = yy[i];
			yy[i] = 0.0;
		  }
	    }

	    if (imode == 1) {
		  if (ilog2(len) <= 0) {
			errwin("Set length not a power of 2");
			return(-1);
		  }
		  fft(xx, yy, len, ilog2(len), !invflag);
	    } else {
		  dft(xx, yy, len, invflag);
	    }

	    tmplen = len / 2;
	    setsetlen(tmpset, tmplen);
	    xx = getsetx(tmpset);             /* xx may point to new address in principle after setsetlen */
	    yy = getsety(tmpset);
 	    switch (load) {
		case 0:
		  delt = x[1] - x[0];
		  T = (len - 1) * delt;
		  for (i = 0; i < tmplen; i++) {
			yy[i] = hypot(xx[i], yy[i]);
			switch (loadx) {
			      case 0:
			        xx[i] = i;
			        break;
			      case 1:
			        /* xx[i] = 2.0 * M_PI * i / ilen; */
			        xx[i] = i / T;
			        break;
			      case 2:
			        if (i == 0) {
				    xx[i] = T + delt;	/* the mean */
			        } else {
				    /* xx[i] = (double) ilen / (double) i; */
				    xx[i] = T / i;
			        }
			        break;
			}
		  }
		  break;
		case 1:
		  delt = x[1] - x[0];
		  T = (x[len - 1] - x[0]);
		  for (i = 0; i < tmplen ; i++) {
			yy[i] = -atan2(yy[i], xx[i]);
			switch (loadx) {
			    case 0:
			      xx[i] = i;
			      break;
			    case 1:
			      /* xx[i] = 2.0 * M_PI * i / ilen; */
			      xx[i] = i / T;
			      break;
			    case 2:
			      if (i == 0) {
				    xx[i] = T + delt;
			      } else {
				    /* xx[i] = (double) ilen / (double) i; */
				    xx[i] = T / i;
			      }
			      break;
			}
		  }
		  break;
	    }
      }
      

      sprintf(reply,"%d %c %s",tmplen,'\0',ptr);

      return(swap_sets(set1,tmpset));
}

int
calc_runa(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int runlen, runtype, len, tmpset, tmplen;
      double *xdata, *ydata, *xtmp, *ytmp;
      char *ptr = command;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d %d",&runlen,&runtype);

/* get data set parameters */
      len = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);

/* get new set to store final results in */
      if ( (tmpset = new_set(len)) == -1)
	    return(-1);
      ytmp = getsety(tmpset);
      xtmp = getsetx(tmpset);

/* perform calculation */      
	switch (runtype) {
	  case 0:
	    runavg(xdata, ydata, xtmp, ytmp, len, runlen);
	    break;
	  case 1:
	    runmedian(xdata, ydata, xtmp, ytmp, len, runlen);
	    break;
  	  case 2:
	    runminmax(xdata, ydata, xtmp, ytmp, len, runlen, 0);
	    break;
	  case 3:
	    runminmax(xdata, ydata, xtmp, ytmp, len, runlen, 1);
	    break;
	  case 4:
	    runstddev(xdata, ydata, xtmp, ytmp, len, runlen);
	    break;
      }

      tmplen = len - runlen + 1;
      setsetlen(tmpset,tmplen);
      sprintf(reply,"%d %c running average",tmplen,'\0');

     return(swap_sets(set1,tmpset));
}

int
calc_reg(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int ideg, iresid, len, tmpset, i;
      char *ptr = command;
      double *xdata, *ydata, *ytmp;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d %d",&ideg,&iresid);
      ptr += strlen(ptr) + 1;

/* get data set parameters */
      len = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);
     
/* get new set to store final results in */
    if ( (tmpset = new_set(len)) == -1)
	  return(-1);
    ytmp = getsety(tmpset);

/* perform calculation */      
 	fitcurve(xdata, ydata, len, ideg, ytmp);
	if (iresid) {
	    for (i = 0; i < len; i++) {
		ytmp[i] = ydata[i] - ytmp[i];
	    }
	}

      sprintf(reply,"%d %c %s",len,'\0',ptr);

     return(swap_sets_y(set1,tmpset));
}

int
calc_diff(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int itype, len, tmpset;
      char *ptr = command;
      double *xdata, *ydata, *xtmp, *ytmp;
      
/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d",&itype);

/* get data set parameters */
      len = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);

/* get new set to store final results in */
      if ( (tmpset = new_set(len-1)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform the calculation */
      switch(itype) {
   	case 0:
	    forwarddiff(xdata, ydata, xtmp, ytmp, len);
	    break;
	case 1:
	    backwarddiff(xdata, ydata, xtmp, ytmp, len);
	    break;
	case 2:
	    centereddiff(xdata, ydata, xtmp, ytmp, len);
	    break;

        default:
	    errwin("calc_xvgr: unrecognised diffence type");
	    return(-1);
      }
      sprintf(reply,"%d %c difference set",len-1,'\0');

      return(swap_sets(set1,tmpset));
}

int
calc_xcor(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int itype, lag, len, tmpset, i;
      char *ptr = command;
      double *ydata1, *ydata2;
      double *xtmp, *ytmp;
      
/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d %d",&itype,&lag);
      ptr += strlen(ptr) + 1;
      
/* get data set parameters */
      len = getsetlen(set1);
      ydata1 = getsety(set1);
      ydata2 = getsety(set2);

/* get new set to store final results in */
      if ( (tmpset = new_set(lag)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform calculation */
      if (crosscorr(ydata1, ydata2, len, lag, itype, xtmp, ytmp) != 0)
	    return(-1);
      for (i = 0; i < lag; i++) {
	    xtmp[i] = i;
      }

      sprintf(reply,"%d %c %s",lag,'\0',ptr);
      return(swap_sets(set1,tmpset));
}

/*
 * splines
 */
int
calc_spline(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
    int    tmpset, len, step, start, stop, i;
    double delx, *b, *c, *d, seval();
    double *xdata, *ydata, *xtmp, *ytmp;
    char   *ptr = command + strlen(command) + 1;

/* get set parameters */
    len = getsetlen(set1);
    xdata = getsetx(set1);
    ydata = getsety(set1);

    sscanf(ptr,"%d %d %d",&step, &start, &stop);
    ptr += strlen(ptr) + 1;
    if (step <= 1 || start >= stop ) {
	  fprintf(stderr,"spline step size too small => %d\n",step);
	  return(-1);
    }
    delx = (stop - start) / (step - 1.0);

/* get new set to store final results in */
    if ( (tmpset = new_set(step)) == -1)
	  return(-1);
    xtmp = getsetx(tmpset);
    ytmp = getsety(tmpset);
    
/* claim memory for calculations */
    b = (double *) calloc(len, sizeof(double));
    c = (double *) calloc(len, sizeof(double));
    d = (double *) calloc(len, sizeof(double));
    if (b == NULL || c == NULL || d == NULL || xtmp == NULL|| ytmp == NULL) {
	  fprintf(stderr,"Not enough memory for splines");
	  cxfree(b);
	  cxfree(c);
	  cxfree(d);
	  return(-1);
    }

/* calculate spline */    
    spline(len, xdata, ydata, b, c, d);
    for (i = 0; i < step; i++) {
	  xtmp[i] = start + i * delx;
	  ytmp[i] = seval(len, xtmp[i], xdata, ydata, b, c, d);
    }

    sprintf(reply,"%d %c %s",step,'\0',ptr);

/* free allocated memory */    
    cxfree(b);
    cxfree(c);
    cxfree(d);
    return(swap_sets(set1,tmpset));
}

int
calc_samp(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int typeno, stepno, tmpset, setlen;
      int i, ierr, npts=0;
      char *ptr = command, exprstr[BUFSIZ];
      double *xdata, *ydata, *xtmp, *ytmp;
      double a, b, c, d;
      extern double result;

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%d %d",&typeno, &stepno);
      ptr += strlen(ptr) + 1;
      strcpy(exprstr,ptr);

/* get data set parameters */
      setlen = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);
      
/* get new set to store final results in */
      if ( (tmpset = new_set(setlen)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform calculation */
      if (typeno == 0) {
	    for (i=0; i<setlen; i += stepno) {
		  xtmp[npts] = xdata[i];
		  ytmp[npts] = ydata[i];
		  npts++;
	    }
      }
      else {
	    fixupstr(exprstr);
	    for (i = 0; i < setlen; i++) {
		  scanner(exprstr, &xdata[i], &ydata[i], 1, &a, &b, &c, &d, 1, i, set1, &ierr);

		  if (ierr) return(-1);
		  if ((int) result) {
			xtmp[npts] = xdata[i];
			ytmp[npts] = ydata[i];
			npts++;
		  }
	    }
      }
      if (npts <= 0) return(-1);
      if (setsetlen(tmpset,npts) == 0) return(-1);
     
      sprintf(reply,"%d %c sample set",npts,'\0');

      return(swap_sets(set1,tmpset));
}

int
calc_digf(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int len1, len2, tmpset, i;
      double *xdata1, *ydata1, *ydata2;
      double *xtmp, *ytmp;
      char   *ptr = command + strlen(command) + 1;
      
/* get data set parameters */
      len1 = getsetlen(set1);
      xdata1 = getsetx(set1);
      ydata1 = getsety(set1);
      len2 = getsetlen(set2);
      ydata2 = getsety(set2);

/* get new set to store final results in */
      if ( (tmpset = new_set(len1)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform calculation */
      filterser(len1,xdata1,ydata1,xtmp,ytmp,ydata2,len2);
      
      sprintf(reply,"%d %c %s",len1,'\0',ptr);
      return(swap_sets(set1,tmpset));
}

int
calc_lconv(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int len1, len2, tmpset, tmplen, i;
      double *ydata1, *ydata2;
      double *xtmp, *ytmp;
      char   *ptr = command + strlen(command) + 1;
      
/* get data set parameters */
      len1 = getsetlen(set1);
      ydata1 = getsety(set1);
      len2 = getsetlen(set2);
      ydata2 = getsety(set2);

/* get new set to store final results in */
      tmplen = len1+len2-1;
      if ( (tmpset = new_set(tmplen)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);

/* perform calculation */
      linearconv(ydata2,ydata1,ytmp,len2,len1);
      for (i=0; i<len1; i++) {
	    xtmp[i] = i;
      }
      
      sprintf(reply,"%d %c %s",tmplen,'\0',ptr);
      return(swap_sets(set1,tmpset));
}
#define MAXPARM 11
int
calc_nonl(fp_in,command,set1,set2)
      FILE   *fp_in;
      char   *command;
      int    set1;
      int    set2;
{
      int len, i, info, errpos, tmpset;
      int npar, loadto, graphto, maxparm;
      double *xdata, *ydata, *xtmp, *ytmp;
      char *ptr = command, fstr[BUFSIZ];
      double tol, a[MAXPARM];

/* decode additional information in command string */      
      ptr += strlen(ptr) + 1;
      sscanf(ptr,"%lg %d %d %d %d",&tol, &npar, &loadto, &graphto, &maxparm);
      ptr += strlen(ptr) + 1;
      {
	    char a_str[BUFSIZ], *ptr2 = ptr;
	    for (i=0; i<maxparm && i<MAXPARM; i++) {
		  sscanf(ptr2,"%s",a_str);
		  ptr2 += strlen(a_str) + 1;
		  a[i] = atof(a_str);
	    }
      }
      ptr += strlen(ptr) + 1;
      strcpy(fstr,ptr);
 

/* get data set parameters */
      len = getsetlen(set1);
      xdata = getsetx(set1);
      ydata = getsety(set1);
     
/* get new set to store final results in */
      if ( (tmpset = new_set(len)) == -1)
	    return(-1);
      xtmp = getsetx(tmpset);
      ytmp = getsety(tmpset);
      if (copyset(set1,tmpset) == -1)
	    return(-1);
      
/* perform calculation */
      lmfit(fstr, len, xtmp, ytmp, npar, a, tol, &info);

      fixupstr(fstr);
      i=0;
      if (info > 0 && info < 4) {
	    switch (loadto) {
		case 0:
		  scanner(fstr, xtmp, ytmp, len, ax, bx, cx, dx, MAXARR, i, tmpset, &errpos); 
		  break;
		case 1:
		  scanner(fstr, xtmp, ytmp, len, ax, bx, cx, dx, MAXARR, i, tmpset, &errpos);
		  for (i=0; i<len; i++) {
			ytmp[i] = ydata[i] - ytmp[i];
		  }
		  break;
	        case 2:
		  break;
	    }
      }
      if (swap_sets(set1,tmpset) == -1)
	    return(-1);

      {
	    char string[BUFSIZ];
	    ptr = string;
	    for (i = 0; i < maxparm; i++) {
		  sprintf(ptr,"%lg %c",a[i],'\0');
		  ptr += strlen(ptr);
	    }
	    sprintf(reply,"%d %c nonlinear fit using %s %c %d %d %d %d %c %s",
		    len,'\0',fstr,'\0',info,loadto,graphto,maxparm,'\0',string);
      }

      return(errpos);
}


/*
 * free and check for NULL pointer
 */
void cxfree(ptr)
    void *ptr;
{
    if (ptr != NULL) {
	free(ptr);
    }
}

/*
 *   write data out to file
 */
void
output_data(command, com_size, x, y, setlen)
      char   *command;
      double *x, *y;
      int    setlen;
      int    com_size;
{
      FILE   *fp;
      char   out_file[L_tmpnam+24];

/* setup output file name */
      sprintf(out_file,"%sCALC_XVGR_%d",P_tmpdir,getpid());
/*      fprintf(stderr,"%s\n",out_file); */

      if (command != NULL) {
	    if ( (fp = fopen(out_file,"w")) == NULL) {
		  errwin("failed to open output data file");
		  return;
	    }
	    if ( fwrite(command, com_size, 1, fp) != 1) {
		  errwin("Write to output data file failed");
		return;
	    }
      }
      else {
	    if ( (fp = fopen(out_file,"a")) == NULL) {
		  errwin("failed to output open data file");
		  return;
	    }
      }
      if ( fwrite((char *) x, sizeof(double), setlen, fp) != setlen) {
	    errwin("Write to output data file failed");
	    return;
      }
      if ( fwrite((char *) y, sizeof(double), setlen, fp) != setlen) {
	    errwin("Write to output data file failed");
	    return;
      }
    (void) fclose(fp);
}
void
errwin(message)
      char *message;
{
      fprintf(stderr,"%s\n",message);
      return;
}

void init_array(a, n)
    int n;
    double **a;
{
    if (*a != NULL) {
	*a = (double *) realloc(*a, n * sizeof(double));
    } else {
	*a = (double *) calloc(n, sizeof(double));
    }
    if (*a == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for array\n");
	exit(1);
    }
}

void init_scratch_arrays(n)
    int n;
{
    init_array(&ax, n);
    init_array(&bx, n);
    init_array(&cx, n);
    init_array(&dx, n);
}

/*
 * forward, backward and centered differences
 */
static void forwarddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h;

    for (i = 1; i < n; i++) {
	resx[i - 1] = x[i - 1];
	h = x[i - 1] - x[i];
	if (h == 0.0) {
	    resy[i - 1] = MBIG;
	    eflag = 1;
	} else {
	    resy[i - 1] = (y[i - 1] - y[i]) / h;
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

static void backwarddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h;

    for (i = 0; i < n - 1; i++) {
	resx[i] = x[i];
	h = x[i + 1] - x[i];
	if (h == 0.0) {
	    resy[i] = MBIG;
	    eflag = 1;
	} else {
	    resy[i] = (y[i + 1] - y[i]) / h;
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

static void centereddiff(x, y, resx, resy, n)
    double x[], y[], resx[], resy[];
int n;

{
    int i, eflag = 0;
    double h1, h2;

    for (i = 1; i < n - 1; i++) {
	resx[i - 1] = x[i];
	h1 = x[i] - x[i - 1];
	h2 = x[i + 1] - x[i];
	if (h1 + h2 == 0.0) {
	    resy[i - 1] = MBIG;
	    eflag = 1;
	} else {
	    resy[i - 1] = (y[i + 1] - y[i - 1]) / (h1 + h2);
	}
    }
    if (eflag) {
	errwin("Warning: infinite slope, check set status before proceeding");
    }
}

apply_window(xx, yy, ilen, type, wind)
    double *xx, *yy;
    int type, wind;
{
    int i;

    for (i = 0; i < ilen; i++) {
	switch (wind) {
	case 1:		/* triangular */
	    if (type != 0) {
		xx[i] *= 1.0 - fabs((i - 0.5 * (ilen - 1.0)) / (0.5 * (ilen - 1.0)));
	    }
	    yy[i] *= 1.0 - fabs((i - 0.5 * (ilen - 1.0)) / (0.5 * (ilen - 1.0)));

	    break;
	case 2:		/* Hanning */
	    if (type != 0) {
		xx[i] = xx[i] * (0.5 - 0.5 * cos(2.0 * M_PI * i / (ilen - 1.0)));
	    }
	    yy[i] = yy[i] * (0.5 - 0.5 * cos(2.0 * M_PI * i / (ilen - 1.0)));
	    break;
	case 3:		/* Welch (from Numerical Recipes) */
	    if (type != 0) {
		xx[i] *= 1.0 - pow((i - 0.5 * (ilen - 1.0)) / (0.5 * (ilen + 1.0)), 2.0);
	    }
	    yy[i] *= 1.0 - pow((i - 0.5 * (ilen - 1.0)) / (0.5 * (ilen + 1.0)), 2.0);
	    break;
	case 4:		/* Hamming */
	    if (type != 0) {
		xx[i] = xx[i] * (0.54 - 0.46 * cos(2.0 * M_PI * i / (ilen - 1.0)));
	    }
	    yy[i] = yy[i] * (0.54 - 0.46 * cos(2.0 * M_PI * i / (ilen - 1.0)));
	    break;
	case 5:		/* Blackman */
	    if (type != 0) {
		xx[i] = xx[i] * (0.42 - 0.5 * cos(2.0 * M_PI * i / (ilen - 1.0)) + 0.08 * cos(4.0 * M_PI * i / (ilen - 1.0)));
	    }
	    yy[i] = yy[i] * (0.42 - 0.5 * cos(2.0 * M_PI * i / (ilen - 1.0)) + 0.08 * cos(4.0 * M_PI * i / (ilen - 1.0)));
	    break;
	case 6:		/* Parzen (from Numerical Recipes) */
	    if (type != 0) {
		xx[i] *= 1.0 - fabs((i - 0.5 * (ilen - 1)) / (0.5 * (ilen + 1)));
	    }
	    yy[i] *= 1.0 - fabs((i - 0.5 * (ilen - 1)) / (0.5 * (ilen + 1)));
	    break;
	}
    }
}

/*
 * log base 2
 */
int ilog2(n)
    int n;
{
    int i = 0;
    int n1 = n;

    while (n1 >>= 1)
	i++;
    if (1 << i != n)
	return -1;
    else
	return i;
}

/*
 *  routines to tidy up after us
 */
void end_proc(ret_code)
      int  ret_code;
{
      (void) unlink(in_file); 
      exit(ret_code);
}
#ifdef SVR4
void sig_fault(int sig)
#else
void sig_fault(sig, code, scp, addr)
      int sig, code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      char buf[BUFSIZ];
      sprintf(buf,"Error during calculation resulted in signal %d",sig);
      errwin(buf);
      end_proc(-2);
}
