/* $Id: setutils.c,v 1.34 1992/08/27 21:53:06 pturner Exp pturner $
 *
 * routines to allocate, manipulate, and return
 * information about sets.
 *
 */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "globals.h"
#include "alerts.h"
#include "defaults.h"
#include "events.h"
#include "graphutils.h"
#include "io.h"
#include "symwin.h"


void sort_set();
void do_moveset();
void killset();
char *my_realloc();

static int default_color[MAXPLOT] = {
				     1, 2, 3, 4, 5, 6, 7, 8, 9, 10
};

int index_set_types[] = {XY, XYDX, XYDY, XYDXDX, XYDYDY, XYDXDY, XYHILO, XYRT, -1};
int index_set_ncols[] = {2, 3, 3, 4, 4, 4, 3, 5, 3, -1};

/*
 * return the string version of the set type
 */
char *set_types(it)
    int it;
{
    char *s = "XY";

    switch (it) {
    case XY:
	s = "xy";
	break;
    case XYDX:
	s = "xydx";
	break;
    case XYDY:
	s = "xydy";
	break;
    case XYDYDY:
	s = "xydydy";
	break;
    case XYDXDX:
	s = "xydxdx";
	break;
    case XYDXDY:
	s = "xydxdy";
	break;
     case XYHILO:
	s = "xyhilo";
	break;
    case XYRT:
	s = "xyrt";
	break;
    }
    return s;
}

/*
 * needed as initplots is called before
 * the number of planes is determined
 */
void setdefaultcolors(gno)
    int gno;
{
    int i;

    for (i = 0; i < MAXPLOT; i++) {
	g[gno].p[i].color = default_color[i];
    }
}

/*
 * allocate arrays for a set of length len.
 */
void allocxy(p, len)
    plotarr *p;
    int len;
{
    int i, ncols;
    int oldlen = p->len;

    switch (p->type) {
    case XY:
	ncols = 2;
	break;
    case XYDX:
    case XYDY:
	ncols = 3;
	break;
    case XYDXDX:
    case XYDYDY:
    case XYDXDY:
    case XYRT:
    case XYX2Y2:
    case XYBOX:
    case XYARC:
    case XYYY:
    case XYXX:
	ncols = 4;
	break;
    case XYHILO:
	ncols = 5;
	break;
    }
    for (i = 0; i < ncols; i++) {
	if (p->ex[i] == NULL) {
	    if ((p->ex[i] = (double *) calloc(len, sizeof(double))) == NULL) {
		fprintf(stderr, "Insufficient memory to allocate for plots\n");
		exit(1);
	    }
	} else {
	    /*LINTED*/
	    if ((p->ex[i] = (double *)
		 my_realloc((char *)p->ex[i], oldlen, len, sizeof(double))) == NULL) {
		    fprintf(stderr, "Insufficient memory to allocate for plots\n");
		    exit(1);
	    }
	}
    }
    p->len = len;
}
/*
void init_array(a, n)
    int n;
    double **a;
{
    if (*a != NULL) {
	*a = (double *) realloc(*a, n * sizeof(double));
    } else {
	*a = (double *) calloc(n, sizeof(double));
    }
    if (*a == NULL) {
	fprintf(stderr, "Insufficient memory to allocate for array\n");
	exit(1);
    }
}

void init_scratch_arrays(n)
    int n;
{
    init_array(&ax, n);
    init_array(&bx, n);
    init_array(&cx, n);
    init_array(&dx, n);
}
*/
/*
 * get the min/max fields of a set
 */
void getsetminmax(gno, setno, x1, x2, y1, y2)
    int gno, setno;
    double *x1, *x2, *y1, *y2;
{
    *x1 = g[gno].p[setno].xmin;
    *x2 = g[gno].p[setno].xmax;
    *y1 = g[gno].p[setno].ymin;
    *y2 = g[gno].p[setno].ymax;
}

/*
 * compute the mins and maxes of a vector x
 */
void minmax(x, n, xmin, xmax, imin, imax)
    double *x;
    int n;
    double *xmin, *xmax;
    int *imin, *imax;
{
    int i;

    *xmin = x[0];
    *xmax = x[0];
    *imin = 1;
    *imax = 1;
    for (i = 1; i < n; i++) {
	if (x[i] < *xmin) {
	    *xmin = x[i];
	    *imin = i + 1;
	}
	if (x[i] > *xmax) {
	    *xmax = x[i];
	    *imax = i + 1;
	}
    }
}

void getsetdxdyminmax(gno, setno, dx1, dx2, dy1, dy2)
    int gno, setno;
    double *dx1, *dx2, *dy1, *dy2;
{
    int itmp;

    if (getcol(gno, setno, 2) != NULL) {
	minmax(getcol(gno, setno, 2), getsetlength(gno, setno), dx1, dx2, &itmp, &itmp);
    }
    if (getcol(gno, setno, 3) != NULL) {
	minmax(getcol(gno, setno, 3), getsetlength(gno, setno), dy1, dy2, &itmp, &itmp);
    }
}

/*
 * compute the mins/maxes and update the appropriate fields of
 * set i.
 */
void updatesetminmax(gno, i)
    int gno, i;
{
    double x1, x2;
    int itmp1, itmp2;

    if (isactive_set(gno, i)) {
	minmax(getx(gno, i), getsetlength(gno, i), &x1, &x2, &itmp1, &itmp2);
	g[gno].p[i].xmin = x1;
	g[gno].p[i].xmax = x2;
	minmax(gety(gno, i), getsetlength(gno, i), &x1, &x2, &itmp1, &itmp2);
	g[gno].p[i].ymin = x1;
	g[gno].p[i].ymax = x2;
    } else {
	g[gno].p[i].xmin = 0.0;
	g[gno].p[i].xmax = 0.0;
	g[gno].p[i].ymin = 0.0;
	g[gno].p[i].ymax = 0.0;
    }
}

void set_point(gno, setn, seti, wx, wy)
    int gno, setn, seti;
    double wx, wy;
{
    g[gno].p[setn].ex[0][seti] = wx;
    g[gno].p[setn].ex[1][seti] = wy;
    updatesetminmax(gno, setn);
}

void get_point(gno, setn, seti, wx, wy)
    int gno, setn, seti;
    double *wx, *wy;
{
    *wx = g[gno].p[setn].ex[0][seti];
    *wy = g[gno].p[setn].ex[1][seti];
}

void setcol(gno, x, setno, len, col)
    double *x;
    int gno, setno, len, col;

{
    g[gno].p[setno].ex[col] = x;
    g[gno].p[setno].len = len;
}

int getncols(gno, setno)
    int gno, setno;
{
    int i = 0;

    while (g[gno].p[setno].ex[i])
	i++;
    return i;
}

void setxy(gno, ex, setno, len, ncols)
    double *ex[];
int gno, setno, len;

{
    int i;

    for (i = 0; i < ncols; i++) {
	g[gno].p[setno].ex[i] = ex[i];
    }
    g[gno].p[setno].len = len;
}

void setlength(gno, i, length)
    int gno, i, length;
{
    allocxy(&g[gno].p[i], length);
}

void copycol(gno, setfrom, setto, col)
    int gno, setfrom, setto, col;
{
    int i, n;
    double *x1, *x2;

    n = g[gno].p[setfrom].len;
    x1 = getcol(gno, setfrom, col);
    x2 = getcol(gno, setto, col);
    for (i = 0; i < n; i++) {
	x2[i] = x1[i];
    }
}

void copycol2(gfrom, setfrom, gto, setto, col)
    int gfrom, setfrom, gto, setto, col;
{
    int i, n;
    double *x1, *x2;

    n = g[gfrom].p[setfrom].len;
    x1 = getcol(gfrom, setfrom, col);
    x2 = getcol(gto, setto, col);
    for (i = 0; i < n; i++) {
	x2[i] = x1[i];
    }
}

/*
 * moveset assumes both sets exist, have their length
 * properly set, and that they are both active
 */
void moveset(gnofrom, setfrom, gnoto, setto)
    int gnofrom, setfrom, gnoto, setto;
{
    int k;
    memcpy(&g[gnoto].p[setto], &g[gnofrom].p[setfrom], sizeof(plotarr));
    for (k = 0; k < MAX_SET_COLS; k++) {
	g[gnofrom].p[setfrom].ex[k] = NULL;
    }
}

/*
 * copyset assumes both sets exist, have their length
 * properly set, and that they are both active
 */
void copyset(gnofrom, setfrom, gnoto, setto)
    int gnofrom, setfrom, gnoto, setto;
{
    int k;
    double *savec[MAX_SET_COLS];
    int len = getsetlength(gnofrom, setfrom);

    for (k = 0; k < MAX_SET_COLS; k++) {
	savec[k] = g[gnoto].p[setto].ex[k];
    }
    memcpy(&g[gnoto].p[setto], &g[gnofrom].p[setfrom], sizeof(plotarr));
    for (k = 0; k < MAX_SET_COLS; k++) {
	g[gnoto].p[setto].ex[k] = savec[k];
	if (g[gnofrom].p[setfrom].ex[k] != NULL && g[gnoto].p[setto].ex[k] != NULL) {
	    memcpy(g[gnoto].p[setto].ex[k], g[gnofrom].p[setfrom].ex[k], len * sizeof(double));
	}
    }
}

/*
 * copy everything but the data
 */
void copysetprops(gnofrom, setfrom, gnoto, setto)
    int gnofrom, setfrom, gnoto, setto;
{
    int k;
    double *savec[MAX_SET_COLS];

    for (k = 0; k < MAX_SET_COLS; k++) {
	savec[k] = g[gnoto].p[setto].ex[k];
    }
    memcpy(&g[gnoto].p[setto], &g[gnofrom].p[setfrom], sizeof(plotarr));
    for (k = 0; k < MAX_SET_COLS; k++) {
	g[gnoto].p[setto].ex[k] = savec[k];
    }
}

/*
 * copy data only
 */
void copysetdata(gnofrom, setfrom, gnoto, setto)
    int gnofrom, setfrom, gnoto, setto;
{
    int k;
    int len = getsetlength(gnofrom, setfrom);
    for (k = 0; k < MAX_SET_COLS; k++) {
	if (g[gnofrom].p[setfrom].ex[k] != NULL && g[gnoto].p[setto].ex[k] != NULL) {
	    memcpy(g[gnoto].p[setto].ex[k], g[gnofrom].p[setfrom].ex[k], len * sizeof(double));
	}
    }
}

/*
 * return the next available set in graph gno
 * ignoring deactivated sets.
 */
int nextset(gno)
    int gno;
{
    int i;

    i = 0;
    for (i = 0; i < g[gno].maxplot; i++) {
	if (!isactive_set(gno, i) && !ishidden_set(gno, i) ) {
	    return (i);
	}
    }
    errwin("Error - no sets available");
    return (-1);
}

/*
 * kill a set
 */
void killset(gno, setno)
    int gno, setno;
{
      int i;
      for (i = 0; i < MAX_SET_COLS; i++) {
	    if (g[gno].p[setno].ex[i] != NULL) {
		  free(g[gno].p[setno].ex[i]);
	    }
      }
      set_default_plotarr(&g[gno].p[setno]);
      g[gno].p[setno].active = OFF;
      g[gno].p[setno].deact = 0;	/* just in case */
}
/*
 * kill a set, but preserve the parameter settings
 */
void softkillset(gno, setno)
    int gno, setno;
{
    int i;

    for (i = 0; i < MAX_SET_COLS; i++) {
	if (g[gno].p[setno].ex[i] != NULL) {
	    free(g[gno].p[setno].ex[i]);
	}
	g[gno].p[setno].ex[i] = NULL;
    }
    g[gno].p[setno].active = OFF;
    g[gno].p[setno].deact = 0;
}

/*
 * activate a set
 */
void activateset(gno, setno)
    int gno, setno;
{
    g[gno].p[setno].active = ON;
    g[gno].p[setno].deact = 0;
}

/*
 * return the active status of a set
 */
int activeset(gno)
    int gno;
{
    int i;

    for (i = 0; i < g[gno].maxplot; i++) {
	if (g[gno].p[i].active == ON) {
	    return (1);
	}
    }
    return (0);
}

/*
 * sort a set - only does type XY
 */
void sort_xy(tmp1, tmp2, up, sorton, stype)
    double *tmp1, *tmp2;
    int up, sorton, stype;
{

    int d, i, j;
    int lo = 0;
    double t1, t2;

    if (sorton == 1) {
	double *ttmp;

	ttmp = tmp1;
	tmp1 = tmp2;
	tmp2 = ttmp;
    }
    up--;

    for (d = up - lo + 1; d > 1;) {
	if (d < 5)
	    d = 1;
	else
	    d = (5 * d - 1) / 11;
	for (i = up - d; i >= lo; i--) {
	    t1 = tmp1[i];
	    t2 = tmp2[i];
	    if (!stype) {
		for (j = i + d; j <= up && (t1 > tmp1[j]); j += d) {
		    tmp1[j - d] = tmp1[j];
		    tmp2[j - d] = tmp2[j];
		}
		tmp1[j - d] = t1;
		tmp2[j - d] = t2;
	    } else {
		for (j = i + d; j <= up && (t1 < tmp1[j]); j += d) {
		    tmp1[j - d] = tmp1[j];
		    tmp2[j - d] = tmp2[j];
		}
		tmp1[j - d] = t1;
		tmp2[j - d] = t2;
	    }
	}
    }
}

/*
 * locate a point and the set the point is in
 */
void findpoint(gno, x, y, xs, ys, setno, loc)
    int gno;
    double x, y;
    double *xs, *ys;
    int *setno, *loc;
{
    double dx = g[gno].w.xg2 - g[gno].w.xg1, dy = g[gno].w.yg2 - g[gno].w.yg1, *xtmp, *ytmp, tmp, tmin = 1.0e307;
    int i, j, len;

    *setno = -1;
    for (i = 0; i < g[gno].maxplot; i++) {
	if (isactive(gno, i)) {
	    xtmp = getx(gno, i);
	    ytmp = gety(gno, i);
	    len = getsetlength(gno, i);
	    for (j = 0; j < len; j++) {
		if ((tmp = hypot((x - xtmp[j]) / dx, (y - ytmp[j]) / dy)) < tmin) {
		    *setno = i;
		    *loc = j + 1;
		    *xs = xtmp[j];
		    *ys = ytmp[j];
		    tmin = tmp;
		}
	    }
	}
    }
}

/*
 * copy a set to another set, if the to set doesn't exist
 * get a new one, if it does, ask if it is okay to overwrite
 */
void do_copyset(gfrom, j1, gto, j2)
    int gfrom, j1, gto, j2;
{
    if (!isactive_graph(gto)) {
	set_graph_active(gto);
    }
    if (!isactive(gfrom, j1)) {
	return;
    }
    if (j1 == j2 && gfrom == gto) {
	return;
    }
    if (isactive(gto, j2)) {
	killset(gto, j2);
    }
    activateset(gto, j2);
    settype(gto, j2, dataset_type(gfrom, j1));
    setlength(gto, j2, getsetlength(gfrom, j1));
    copyset(gfrom, j1, gto, j2);
    sprintf(buf, "copy of set %d", j1);
/*    setcomment(gto, j2, buf); */
    updatesetminmax(gto, j2);
/*    update_set_status(gto, j2); */
}

/*
 * move a set to another set, in possibly another graph
 */
void do_moveset(gfrom, j1, gto, j2)
    int gfrom, j1, gto, j2;
{
    if (!isactive_graph(gto)) {
	set_graph_active(gto);
    }
    if (!isactive(gfrom, j1)) {
	return;
    }
    if (j1 == j2 && gto == gfrom) {
	return;
    }
    if (isactive(gto, j2)) {
	killset(gto, j2);
    }
    moveset(gfrom, j1, gto, j2);
    updatesymbols(gto, j2); 
    updatesymbols(gfrom, j1); 
    updatelegendstr(gto); 
    updatesetminmax(gto, j2);
/*    update_set_status(gto, j2); */
    killset(gfrom, j1);
/*    update_set_status(gfrom, j1); */
}

/*
 * swap a set with another set
 */
void do_swapset(gfrom, j1, gto, j2)
    int gfrom, j1, j2, gto;
{
    plotarr p;

    if (j1 == j2 && gto == gfrom) {
	errwin("Set from and set to are the same");
	return;
    }
    memcpy(&p, &g[gto].p[j1], sizeof(plotarr));
    memcpy(&g[gto].p[j1], &g[gfrom].p[j2], sizeof(plotarr));
    memcpy(&g[gfrom].p[j2], &p, sizeof(plotarr));
    updatesetminmax(gfrom, j1);
    updatesymbols(gfrom, j1); 
    updatelegendstr(gfrom); 
/*    update_set_status(gfrom, j1); */
    updatesetminmax(gto, j2);
    updatesymbols(gto, j2); 
    updatelegendstr(gto); 
/*    update_set_status(gto, j2); */
    drawgraph2(-1);
}

/*
 * activate a set and set its length
 */
void do_activateset(gno, setno, len)
    int gno, setno, len;
{
    if (isactive(gno, setno)) {
	sprintf(buf, "Set %d already active", setno);
	errwin(buf);
	return;
    }
    if (len <= 0) {
	sprintf(buf, "Improper set length = %d", len);
	errwin(buf);
	return;
    }
    activateset(gno, setno);
    setlength(gno, setno, len);
    updatesetminmax(gno, setno);
/*    update_set_status(gno, setno); */
}

/*
 * activate a set and set its length
 */
void do_activate(setno, type, len)
    int setno, len, type;
{
    type = index_set_types[type];
    if (isactive(cg, setno)) {
	sprintf(buf, "Set %d already active", setno);
	errwin(buf);
	return;
    }
    if (len <= 0) {
	sprintf(buf, "Improper set length = %d", len);
	errwin(buf);
	return;
    }
    activateset(cg, setno);
    settype(cg, setno, type);
    setlength(cg, setno, len);
    updatesetminmax(cg, setno);
/*    update_set_status(cg, setno); */
}

/*
 * de-activate a set
 */
void do_deactivate(gno, setno)
    int gno, setno;
{
/*    set_prop(gno, SET, SETNUM, setno, ACTIVE, OFF, 0); */
    g[gno].p[setno].deact = ON;
/*    update_set_status(gno, setno); */
    drawgraph2(-1);
}
void do_showfit_peaks()
{
      int i, update = 0;
      for (i=0; i < g[cg].maxplot; i++) {
	    if ( isactive(cg,i)) 
		  if ((g[cg].p[i].spec == BUFFIT_DATA) && ishidden_set(cg,i)) {
			    g[cg].p[i].deact = HIDE;
			    update = 1;
		      }
      }
      if (update) drawgraph2(-1);
      return;
}
void do_hidefit_peaks()
{
      int i, update = 0;
      for (i=0; i < g[cg].maxplot; i++) {
	    if ( isactive(cg,i)) 
		  if ((g[cg].p[i].spec == BUFFIT_DATA) && (g[cg].p[i].deact == HIDE)) {
			    hide_set(cg,i);
			    update = 1;
		      }
      }
      if (update) drawgraph2(-1);
      return;
}
/*
 * re-activate a set
 */
void do_reactivate(gno, setno)
    int gno, setno;
{
/*    set_prop(gno, SET, SETNUM, setno, ACTIVE, ON, 0); */
    if (! ishidden_set(gno,setno) || g[gno].p[setno].ex[0] == NULL) return;
    g[gno].p[setno].active = ON;
    g[gno].p[setno].deact = 0;
/*    update_set_status(gno, setno); */
    drawgraph2(-1);
}

/*
 * change the type of a set
 */
void do_changetype(setno, type)
    int setno, type;
{
    type = index_set_types[type];
    settype(cg, setno, type);
    setlength(cg, setno, getsetlength(cg, setno));
    updatesetminmax(cg, setno);
/*    update_set_status(cg, setno); */
}

/*
 * set the length of an active set - contents are destroyed
 */
void do_setlength(setno, len)
    int setno, len;
{
    if (!isactive(cg, setno)) {
	sprintf(buf, "Set %d not active", setno);
	errwin(buf);
	return;
    }
    if (len <= 0) {
	sprintf(buf, "Improper set length = %d", len);
	errwin(buf);
	return;
    }
    setlength(cg, setno, len);
    updatesetminmax(cg, setno);
/*    update_set_status(cg, setno); */
}

/*
 * copy a set to another set, if the to set doesn't exist
 * get a new one, if it does, ask if it is okay to overwrite
 */
void do_copy(j1, gfrom, j2, gto)
    int j1, j2, gfrom, gto;
{
    if (!isactive(gfrom, j1)) {
	sprintf(buf, "Set %d not active", j1);
	errwin(buf);
	return;
    }
    gto--;
    if (gto == -1) {
	gto = cg;
    }
    if (!isactive_graph(gto)) {
	set_graph_active(gto);
    }
    if (j1 == j2 - 1 && gfrom == gto) {
	errwin("Set from and set to are the same");
	return;
    }
    /* select next set */
    if (j2 == 0) {
	if ((j2 = nextset(gto)) != -1) {
	    activateset(gto, j2);
	    settype(gto, j2, dataset_type(gfrom, j1));
	    setlength(gto, j2, getsetlength(gfrom, j1));
	} else {
	    return;
	}
    }
    /* use user selected set */
    else {
	j2--;			/* extra item in this cycle */
	if (isactive(gto, j2)) {
	    sprintf(buf, "Set %d active, overwrite?", j2);
	    if (!yesno(buf, "", "YES", "NO")) {
		return;
	    }
	    killset(gto, j2);
	}
	activateset(gto, j2);
	settype(gto, j2, dataset_type(gfrom, j1));
	setlength(gto, j2, getsetlength(gfrom, j1));
    }
    copyset(gfrom, j1, gto, j2);
    sprintf(buf, "copy of set %d", j1);
/*    setcomment(gto, j2, buf); */
    updatesetminmax(gto, j2);
/*    update_set_status(gto, j2); */
    drawgraph2(-1);
}

/*
 * move a set to another set, if the to set doesn't exist
 * get a new one, if it does, ask if it is okay to overwrite
 */
void do_move(j1, gfrom, j2, gto)
    int j1, gfrom, j2, gto;
{
    if (!isactive(gfrom, j1)) {
	sprintf(buf, "Set %d not active", j1);
	errwin(buf);
	return;
    }
    gto--;
    if (gto == -1) {
	gto = cg;
    }
    if (!isactive_graph(gto)) {
	set_graph_active(gto);
    }
    if (j2 < 0) {
	if ((j2 = nextset(gto)) == -1) {
	    return;
	}
    }
    if (j1 == j2 && gto == gfrom) {
	errwin("Set from and set to are the same");
	return;
    }
    if (isactive(gto, j2)) {
	sprintf(buf, "Set %d active, overwrite?", j2);
	if (!yesno(buf, "", "YES", "NO")) {
	    return;
	}
	killset(gto, j2);
    }
    moveset(gfrom, j1, gto, j2);
    updatesymbols(gto, j2); 
    updatesymbols(gfrom, j1); 
    updatelegendstr(gto); 
    updatesetminmax(gto, j2);
/*    update_set_status(gto, j2); */
    killset(gfrom, j1);
/*    update_set_status(gfrom, j1); */
    drawgraph2(-1);
}

/*
 * swap a set with another set
 */
void do_swap(j1, gfrom, j2, gto)
    int j1, gfrom, j2, gto;
{
    gfrom--;
    if (gfrom == -1) {
	gfrom = cg;
    }
    gto--;
    if (gto == -1) {
	gto = cg;
    }
    if (j1 == j2 && gfrom == gto) {
	errwin("Set from and set to are the same");
	return;
    }
    do_swapset(gfrom, j1, gto, j2);
}

/*
 * add a point to setno
 */
void add_point(gno, setno, px, py, type)
    int gno, setno, type;
    double px, py;
{
    int len = 0;
    double *x, *y;

    if (isactive(gno, setno)) {
	x = getx(gno, setno);
	y = gety(gno, setno);
	len = getsetlength(gno, setno);
	x = (double *) realloc(x, (len + 1) * sizeof(double));
	y = (double *) realloc(y, (len + 1) * sizeof(double));
	setcol(gno, x, setno, len + 1, 0);
	setcol(gno, y, setno, len + 1, 1);
	x[len] = px;
	y[len] = py;
    } else {
	g[gno].active = ON;
	activateset(gno, setno);
	g[gno].p[setno].type = type;
	allocxy(&g[gno].p[setno], 1);
	x = getx(gno, setno);
	y = gety(gno, setno);
	x[0] = px;
	y[0] = py;
    }
    updatesetminmax(gno, setno);
}

/*
 * kill a set
 */
void do_kill(setno, soft)
    int setno, soft;
{
    int redraw = 0, i;

    if (setno == g[cg].maxplot) {
	for (i = 0; i < g[cg].maxplot; i++) {
	    if (isactive(cg, i)) {
		if (soft) {
		    softkillset(cg, i);
		} else {
		    killset(cg, i);
		}
		redraw = 1;
/*		update_set_status(cg, i); */
	    }
	}
	if (redraw) {
	    drawgraph2(-1);
	} else {
	    errwin("No sets to kill");
	}
    } else {
	if (!isactive(cg, setno)) {
	    sprintf(buf, "Set %d already dead", setno);
	    errwin(buf);
	    return;
	} else {
	    if (soft) {
		softkillset(cg, setno);
	    } else {
		killset(cg, setno);
	    }
/*	    update_set_status(cg, setno); */
	    drawgraph2(-1);
	}
    }
}

/*
 * kill all active sets
 */
void do_flush()
{
    int i;

    if (yesno("Flush all active sets, are you sure? ", "", "YES", "NO")) {
	for (i = 0; i < g[cg].maxplot; i++) {
	    if (isactive(cg, i)) {
		killset(cg, i);
/*		update_set_status(cg, i); */
	    }
	}
	drawgraph2(-1);
    }
}

/*
 * sort sets, only works on sets of type XY
 */
void do_sort(setno, sorton, stype)
    int setno, sorton, stype;
{
    int i;

    if (setno == -1) {
	for (i = 0; i < g[cg].maxplot; i++) {
	    if (isactive(cg, i)) {
		sort_set(i, sorton, stype);
	    }
	}
    } else {
	if (!isactive(cg, setno)) {
	    sprintf(buf, "Set %d not active", setno);
	    errwin(buf);
	    return;
	} else {
	    sort_set(setno, sorton, stype);
	}
    }
    drawgraph2(-1);
}

void sort_set(setno, sorton, stype)
    int setno, sorton, stype;
{
    int up;

    up = getsetlength(cg, setno);
    if (up < 2) {
	return;
    }
    sort_xy(getx(cg, setno), gety(cg, setno), up, sorton, stype);
}

/*
 * kill nearest set
 */
void do_kill_nearest()
{
     set_action(0);
    set_action(KILL_NEAREST);
}

/*
 * copy nearest set
 */
void do_copy_nearest()
{
     set_action(0);
    set_action(COPY_NEAREST1ST);
}

/*
 * move nearest set
 */
void do_move_nearest()
{
     set_action(0);
    set_action(MOVE_NEAREST1ST);
}

/*
 * deactivate the nearest set
 */
void do_deactivate_nearest()
{
     set_action(0);
    set_action(DEACTIVATE_NEAREST);
}

/*
 * delete nearest set
 */
void do_delete_nearest()
{
     set_action(0);
    set_action(DELETE_NEAREST1ST);
}

char *my_realloc(ptr, old_size, new_size, elsize)
      char     *ptr;
      int      old_size;
      int      new_size;
      int      elsize;
{
      char *new_ptr;
      int  copy_size = (new_size > old_size) ? old_size : new_size;
      if (ptr == NULL) return(NULL);
      if ( (new_ptr = calloc(new_size, elsize)) == NULL) {
	    fprintf(stderr,"reallocation of spectrum memory storage failed\n");
	    return(NULL);
      }
      (void) memcpy(new_ptr, ptr, (unsigned) copy_size*elsize);
      (void) free(ptr);
      return(new_ptr);
}

/*
 * drop points from a set
 */
void droppoints(gno, setno, endno, dist)
    int gno, setno, endno, dist;
{
    double *x;
    int i, j, len, ncols;

    len = getsetlength(gno, setno);
    ncols = getncols(gno, setno);
    for (j = 0; j < ncols; j++) {
	x = getcol(gno, setno, j);
	for (i = endno + 1; i < len; i++) {
	    x[i - dist] = x[i];
	}
    }
    setlength(gno, setno, len - dist);
}
/*
 * drop points from an active set
 */
void do_drop_points(setno, startno, endno)
    int setno, startno, endno;
{
    int dist;

    if (!isactive(cg, setno)) {
	sprintf(buf, "Set %d not active", setno);
	errwin(buf);
	return;
    }
    dist = endno - startno + 1;
    if (startno < 0) {
	errwin("Start # < 1");
	return;
    }
    if (endno >= getsetlength(cg, setno)) {
	errwin("Ending # > set length");
	return;
    }
    if (startno > endno) {
	errwin("Starting # > ending #");
	return;
    }
    if (dist == getsetlength(cg, setno)) {
	errwin("# of points to drop = set length, use kill");
	return;
    }
    droppoints(cg, setno, endno, dist);
    updatesetminmax(cg, setno);
/*    update_set_status(cg, setno); */
    drawgraph2(-1);
}

/*
 * write out a set
 */
/*ARGSUSED*/
void do_writesets(gno, setno, imbed, fn, format)
    int gno, setno, imbed;
    char *fn, *format;
{
    int i, j, k, n, which_graph = gno, save_cg = cg, start, stop, set_start, set_stop;
    FILE *cp;
    double *x, *y, *dx, *dy, *dz;

    if (!fn[0]) {
	errwin("Define file name first");
	return;
    }
    if (fexists(fn)) {
	return;
    }
    if ((cp = fopen(fn, "w")) == NULL) {
	char s[192];

	sprintf(s, "Unable to open file %s", fn);
	errwin(s);
	return;
    }
    if (which_graph == MAXGRAPH) {
	start = 0;
	stop = MAXGRAPH - 1;
    } else if (which_graph == -1) {
	start = cg;
	stop = cg;
    } else {
	start = which_graph;
	stop = which_graph;
    }
/*    if (imbed) {
	if (start != stop) {
	    putparms(-1, cp, imbed);
	} else {
	    putparms(start, cp, imbed);
	}
    } */
    for (k = start; k <= stop; k++) {
	if (isactive_graph(k)) {
	    if (start != stop) {
		fprintf(cp, "@WITH G%1d\n", k);
		fprintf(cp, "@G%1d ON\n", k);
	    }
	    if (setno == -1) {
		set_start = 0;
		set_stop = g[cg].maxplot - 1;
	    } else {
		set_start = setno;
		set_stop = setno;
	    }
	    for (j = set_start; j <= set_stop; j++) {
		if (isactive(k, j)) {
		    fprintf(cp, "@TYPE %s\n", set_types(dataset_type(k, j)));
		    x = getx(k, j);
		    y = gety(k, j);
		    n = getsetlength(k, j);
		    switch (dataset_type(k, j)) {
		    case XY:
			for (i = 0; i < n; i++) {
			    fprintf(cp, format, x[i], y[i]);
			    fputc('\n', cp);
			}
			break;
		    case XYDX:
		    case XYDY:
		    case XYZ:
		    case XYRT:
			dx = getcol(k, j, 2);
			for (i = 0; i < n; i++) {
			    fprintf(cp, "%g %g %g", x[i], y[i], dx[i]);
			    fputc('\n', cp);
			}
			break;
		    case XYDXDX:
		    case XYDYDY:
		    case XYDXDY:
			dx = getcol(k, j, 2);
			dy = getcol(k, j, 3);
			for (i = 0; i < n; i++) {
			    fprintf(cp, "%g %g %g %g", x[i], y[i], dx[i], dy[i]);
			    fputc('\n', cp);
			}
			break;
		    case XYHILO:
			dx = getcol(k, j, 2);
			dy = getcol(k, j, 3);
			dz = getcol(k, j, 4);
			for (i = 0; i < n; i++) {
			    fprintf(cp, "%g %g %g %g %g", x[i], y[i], dx[i], dy[i], dz[i]);
			    fputc('\n', cp);
			}
			break;
		    }
		    fprintf(cp, "&\n");
		}
	    }
	}
    }
    fclose(cp);
    cg = save_cg;
}

/*
 * pack all sets leaving no gaps in the set structure
 */
void packsets(gno)
    int gno;
{
    int i, j, k;

    i = 0;
    for (i = 0; i < g[gno].maxplot; i++) {
	if (isactive_set(gno, i)) {
	    j = 0;
	    while (j < i) {
		if (!isactive_set(gno, j)) {
		    memcpy(&g[gno].p[j], &g[gno].p[i], sizeof(plotarr));
		    for (k = 0; k < MAX_SET_COLS; k++) {
			g[gno].p[i].ex[k] = NULL;
		    }
		    killset(gno, i);
		    updatesymbols(gno, j);
		    updatesymbols(gno, i);
		    updatelegendstr(gno);
		    updatesetminmax(gno, j);
		    updatesetminmax(gno, i);
/*		    update_set_status(gno, j);
		    update_set_status(gno, i); */
		}
		j++;
	    }
	}
    }
}

/*
 * action proc for menu item
 */
void do_packsets()
{
     packsets(cg);
}
