/* 
 * ``coord.c''  Routines for Coordinate Reconstruction     SPGC
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "evsubs.h"

struct det_coord *dcp = NULL;

/*
 * Initialise detector coordinates
 */
int coord_init(int n){
    /*create space for dcp[1 to n detectors] structure array*/
    
    if (dcp != NULL)
	free(dcp);
    
    if ((dcp = (struct det_coord *) 
	 calloc(n+1, sizeof(struct det_coord))) == NULL)
    {
	fprintf(stderr, "coord: Unable to allocate memory"
		" for detector coord structure.\n");
	return ABORT;
    }
    return OK;
}

static int coord_readxyz(FILE *fp, struct xyz *r)
{
    char buf[SUBS_BUFSIZE];

    if (mygetline(buf, SUBS_BUFSIZE, fp) == NULL)
    {
	fprintf(stderr, "coord: Unexpected end of coords file.\n");
	return ABORT;
    }
    if(sscanf(buf, " %lf %lf %lf", &(r->x), &(r->y), &(r->z)) != 3) {
	fprintf(stderr,"coord: Error! Expected coords xyz got \"%s\"\n",buf);
	return ABORT;
    }
    
    return OK;
}
    

/*
 * Process Detector coordinate file
 */
int coord_det(int i, FILE *fp)
{
    struct xy xydat;
    struct vector hitvec;
    
    if (coord_readxyz(fp, &(dcp[i].ra)) || coord_readxyz(fp, &(dcp[i].rb)) ||
	coord_readxyz(fp, &(dcp[i].rc)) || coord_readxyz(fp, &(dcp[i].rd)))
	return SKIP;

    /*calculate vectors & detector centre coordinates*/
    xydat.x = 0.0;
    xydat.y = 0.0;
    coord(i, &xydat, &hitvec);
    dcp[i].r0.x = vector_x(&hitvec);
    dcp[i].r0.y = vector_y(&hitvec);
    dcp[i].r0.z = vector_z(&hitvec);
    dcp[i].Xp.x = dcp[i].rb.x - dcp[i].ra.x;
    dcp[i].Xp.y = dcp[i].rb.y - dcp[i].ra.y;
    dcp[i].Xp.z = dcp[i].rb.z - dcp[i].ra.z;
    dcp[i].Yp.x = dcp[i].rd.x - dcp[i].ra.x;
    dcp[i].Yp.y = dcp[i].rd.y - dcp[i].ra.y;
    dcp[i].Yp.z = dcp[i].rd.z - dcp[i].ra.z;   
    dcp[i].n.x = dcp[i].Xp.y*dcp[i].Yp.z - dcp[i].Xp.z*dcp[i].Yp.y;
    dcp[i].n.y = dcp[i].Xp.z*dcp[i].Yp.x - dcp[i].Xp.x*dcp[i].Yp.z;
    dcp[i].n.z = dcp[i].Xp.x*dcp[i].Yp.y - dcp[i].Xp.y*dcp[i].Yp.x;	
    dcp[i].Xp2 =  dcp[i].Xp.x*dcp[i].Xp.x + dcp[i].Xp.y*dcp[i].Xp.y +
	dcp[i].Xp.z*dcp[i].Xp.z;
    dcp[i].Yp2 =  dcp[i].Yp.x*dcp[i].Yp.x + dcp[i].Yp.y*dcp[i].Yp.y + 
	dcp[i].Yp.z*dcp[i].Yp.z;  
    dcp[i].Xp0 = dcp[i].r0.x*dcp[i].Xp.x + dcp[i].r0.y*dcp[i].Xp.y + 
	dcp[i].r0.z*dcp[i].Xp.z;
    dcp[i].Yp0 = dcp[i].r0.x*dcp[i].Yp.x + dcp[i].r0.y*dcp[i].Yp.y + 
	dcp[i].r0.z*dcp[i].Yp.z;
    
    return OK;
}
/*
 * Generate coordinates for ancillary detector
 */
int coord_andet(int i, FILE *fp)
{
    char buf[SUBS_BUFSIZE];
    double thx,thy,l,h,r;
    
    if (mygetline(buf, SUBS_BUFSIZE, fp) == NULL)
    {
	fprintf(stderr, "Unexpected end of coords file.\n");
	return ABORT;
    }
    if( sscanf(buf, " %lf %lf %lf %lf %lf",&thx,&thy,&l,&h,&r) !=5){
	fprintf(stderr,"Error! Expected "
		"'thx thy l h r'\n"
		"got \"%s\".\n", buf);
	return ABORT;
    }
    
    dcp[i].r0.x = r*cos(thy*RAD)*sin(thx*RAD);
    dcp[i].r0.y = r*sin(thy*RAD);
    dcp[i].r0.z = r*cos(thy*RAD)*cos(thx*RAD);
    
    dcp[i].Xp.x = l*cos(thx*RAD);
    dcp[i].Xp.y = 0;
    dcp[i].Xp.z = -l*sin(thx*RAD);
    dcp[i].Yp.x = 0;
    dcp[i].Yp.y = h*cos(thy*RAD);
    dcp[i].Yp.z = h*sin(thy*RAD);
    
    dcp[i].ra.x = dcp[i].r0.x - dcp[i].Xp.x/2 - dcp[i].Yp.x/2;
    dcp[i].ra.y = dcp[i].r0.y - dcp[i].Xp.y/2 - dcp[i].Yp.y/2;
    dcp[i].ra.z = dcp[i].r0.z - dcp[i].Xp.z/2 - dcp[i].Yp.z/2;
    dcp[i].rb.x = dcp[i].r0.x + dcp[i].Xp.x/2 - dcp[i].Yp.x/2;
    dcp[i].rb.y = dcp[i].r0.y + dcp[i].Xp.y/2 - dcp[i].Yp.y/2;
    dcp[i].rb.z = dcp[i].r0.z + dcp[i].Xp.z/2 - dcp[i].Yp.z/2;
    dcp[i].rc.x = dcp[i].r0.x + dcp[i].Xp.x/2 + dcp[i].Yp.x/2;
    dcp[i].rc.y = dcp[i].r0.y + dcp[i].Xp.y/2 + dcp[i].Yp.y/2;
    dcp[i].rc.z = dcp[i].r0.z + dcp[i].Xp.z/2 + dcp[i].Yp.z/2;
    dcp[i].rd.x = dcp[i].r0.x - dcp[i].Xp.x/2 + dcp[i].Yp.x/2;
    dcp[i].rd.y = dcp[i].r0.y - dcp[i].Xp.y/2 + dcp[i].Yp.y/2;
    dcp[i].rd.z = dcp[i].r0.z - dcp[i].Xp.z/2 + dcp[i].Yp.z/2;   

    dcp[i].n.x = dcp[i].Xp.y*dcp[i].Yp.z - dcp[i].Xp.z*dcp[i].Yp.y;
    dcp[i].n.y = dcp[i].Xp.z*dcp[i].Yp.x - dcp[i].Xp.x*dcp[i].Yp.z;
    dcp[i].n.z = dcp[i].Xp.x*dcp[i].Yp.y - dcp[i].Xp.y*dcp[i].Yp.x;	
    dcp[i].Xp2 =  dcp[i].Xp.x*dcp[i].Xp.x + dcp[i].Xp.y*dcp[i].Xp.y +
	dcp[i].Xp.z*dcp[i].Xp.z;
    dcp[i].Yp2 =  dcp[i].Yp.x*dcp[i].Yp.x + dcp[i].Yp.y*dcp[i].Yp.y + 
	dcp[i].Yp.z*dcp[i].Yp.z;
    dcp[i].Xp0 = dcp[i].r0.x*dcp[i].Xp.x + dcp[i].r0.y*dcp[i].Xp.y + 
	dcp[i].r0.z*dcp[i].Xp.z;
    dcp[i].Yp0 = dcp[i].r0.x*dcp[i].Yp.x + dcp[i].r0.y*dcp[i].Yp.y + 
	dcp[i].r0.z*dcp[i].Yp.z;

    return OK;
}


/*
 * Calculate event coordinates for a hit on a detector
 */
int coord(int dt, struct xy *cp, struct vector *hp)
{    
    double a;
    struct det_coord *dp = dcp + dt;

    /*detector to lab coordinate transformation*/

    a = (0.5-cp->x-cp->y);
    
    vector_set_xyz(
	hp,
	a*dp->ra.x + cp->x*dp->rb.x + 0.5*dp->rc.x + cp->y*dp->rd.x,
	a*dp->ra.y + cp->x*dp->rb.y + 0.5*dp->rc.y + cp->y*dp->rd.y,
	a*dp->ra.z + cp->x*dp->rb.z + 0.5*dp->rc.z + cp->y*dp->rd.z);
    
    return OK;
}

/*
 * Calculate detector hit coordinates from particle trajectory.
 * The vector passed in hp is preserved.
 */
int det_coord(int i, struct xy *cp, struct vector *hp)
{
    double gamma1,gamma2;
    double x, y, z;
    
    x = vector_x(hp);
    y = vector_y(hp);
    z = vector_z(hp);

    gamma1 = 1.0/vector_r(hp);

    x *= gamma1;
    y *= gamma1;
    z *= gamma1;

    gamma1 = dcp[i].r0.x*dcp[i].n.x + dcp[i].r0.y*dcp[i].n.y + 
	      dcp[i].r0.z*dcp[i].n.z;
    gamma2 = x*dcp[i].n.x + y*dcp[i].n.y + z*dcp[i].n.z;
    
    if ((gamma1 < 0) != (gamma2 < 0) || gamma2 == 0)
	return 0;

    gamma1 /= gamma2;
    
    x *= gamma1;
    y *= gamma1;
    z *= gamma1;

    cp->x = x*dcp[i].Xp.x + y*dcp[i].Xp.y + z*dcp[i].Xp.z - dcp[i].Xp0;
    cp->y = x*dcp[i].Yp.x + y*dcp[i].Yp.y + z*dcp[i].Yp.z - dcp[i].Yp0;
    
    if (fabs(cp->x) <= 0.5*dcp[i].Xp2 && fabs(cp->y) <= 0.5*dcp[i].Yp2)
	return(i);
    else
	return(0);
}


/*
 * Return which detector was hit for a th,phi trajectory
 */
int ndet_hit(int ndet, double th, double phi)
{
    int i;
    for(i=1; i <ndet+1; i++){
	if(det_hit(i, th, phi))
	    return(i);
    }
    return(0);
}
/*
 * Check if hit a particular detector
 */
int det_hit(int i, double th, double phi)
{
    struct xyz rhat,r;
    double gamma,gamma1,gamma2,xd,yd;
    
    rhat.x = sin(th*RAD);
    rhat.y = rhat.x*sin(phi*RAD);
    rhat.x *= cos(phi*RAD);
    rhat.z = cos(th*RAD);
    
    gamma1 = (dcp[i].r0.x*dcp[i].n.x + dcp[i].r0.y*dcp[i].n.y + 
	      dcp[i].r0.z*dcp[i].n.z);
    gamma2= (rhat.x*dcp[i].n.x + rhat.y*dcp[i].n.y + rhat.z*dcp[i].n.z);
    
    if ((gamma1 < 0) != (gamma2 < 0))
	return(0);
    
    gamma = gamma1/gamma2;
    
    r.x = gamma*rhat.x;
    r.y = gamma*rhat.y;
    r.z = gamma*rhat.z;
    
    xd = (r.x*dcp[i].Xp.x + r.y*dcp[i].Xp.y +r.z*dcp[i].Xp.z - dcp[i].Xp0)/
	dcp[i].Xp2;
    yd = (r.x*dcp[i].Yp.x + r.y*dcp[i].Yp.y +r.z*dcp[i].Yp.z - dcp[i].Yp0)/
	dcp[i].Yp2;
    
    if (fabs(xd) <= 0.5 && fabs(yd) <= 0.5)
	return(i);
    else
	return(0);
}

int *hitbins = NULL;
static double thbinsize, phbinsize;
static int thextent, phextent;

void freehitbins(void)
{
    if (hitbins != NULL)
	free(hitbins);

    hitbins = NULL;
}

int inithitbins(double ts, double ps)
{
    thbinsize = ts;
    phbinsize = ps;

    freehitbins();

    thextent = (int) (180*ts+0.5);
    phextent = (int) (360*ps+0.5);

    if ((hitbins = (int *) calloc(thextent*phextent, sizeof(int)))
	== NULL)
    {
	perror("inithitbins");
	return 0;
    }

    return 1;
}

/*
 * Returns pointer to a bin in th,phi containing hit info
 */
int *gethitbin(double theta, double phi)
{
    int it, ip;

    if (hitbins == NULL)
	if (!inithitbins(2.0, 2.0))
	    return NULL;

    it = theta*thbinsize;
    ip = phi*phbinsize;
    
    if (it >= thextent)
	it = thextent-1;

    if (ip >= phextent)
	ip = phextent-1;

    return(hitbins + it*phextent + ip);
}
