/*
 * ''corr''      Angular Correlations                      Steve Chappell
 *
 *   General Subroutine to reconstruct angular correlations for
 *   decay of a resonant particle into two breakup fragments
 *   given the position of the virtual resonant particle
 *   in the event array
 *   rp* -> Heavy Ion (hi) + Light Ion (li)
 
 * Notes
 
 *   In Polar Co-ords  (hor,vert,beam)=(x,y,z)
 *   In Fixed Axial Co-ords          (xa,ya,za)
 *   In Basel Co-ords                (xb,yb,zb)
 
 * Updates
 *   18th Dec 97: Angles converted to positive range 0 -> 360
 *                Basel.phis 0->180
 *                Bug Fix in axial->Basel transform
 */

#include <stdio.h>
#include <math.h>
#include "evsubs.h"

struct coords pspace, *psp = &pspace;

int corr(particle *vrp)
{
   
    double pcm;
    double p_li_azy,p_hi_azy,p_vrp_azy,dths,tha_li,tha_hi;
    particle *evhi, *evli;
    double mli, mhi;

    /*fill out phase space structure for polar co-ord frame*/    
    /*(Nb.Particles hi,li in lab, vrp in c.m. frame)*/
    evhi = ev_hi(vrp);
    evli = ev_li(vrp); 
    
    psp->polar.p_li.x=ev_p_x(evli);
    psp->polar.p_li.y=ev_p_y(evli);
    psp->polar.p_li.z=ev_p_z(evli);
    mli = ev_m(evli);

    psp->polar.p_hi.x=ev_p_x(evhi);
    psp->polar.p_hi.y=ev_p_y(evhi);
    psp->polar.p_hi.z=ev_p_z(evhi);
    mhi = ev_m(evhi);

    /*Phase Space Analysis ...*/

    psp->polar.p_vrp.x=ev_p_x(vrp);
    psp->polar.p_vrp.y=ev_p_y(vrp);
    psp->polar.p_vrp.z=ev_p_z(vrp)-ev_m(vrp)*vrp->head->reac->vcm;
    pcm=sqrt(psp->polar.p_vrp.x*psp->polar.p_vrp.x+
	     psp->polar.p_vrp.y*psp->polar.p_vrp.y+
	     psp->polar.p_vrp.z*psp->polar.p_vrp.z);

    psp->polar.vrel.x=psp->polar.p_li.x/mli-psp->polar.p_hi.x/mhi;
    psp->polar.vrel.y=psp->polar.p_li.y/mli-psp->polar.p_hi.y/mhi;
    psp->polar.vrel.z=psp->polar.p_li.z/mli-psp->polar.p_hi.z/mhi;

    psp->polar.ths=(acos(psp->polar.p_vrp.z/pcm))*DEG;

    psp->polar.phis=((psp->polar.p_vrp.y >= 0) ? 1 : -1)*
	             (acos(psp->polar.p_vrp.x/
                     (sqrt(psp->polar.p_vrp.x*psp->polar.p_vrp.x+
			   psp->polar.p_vrp.y*psp->polar.p_vrp.y))))*DEG
			       +((psp->polar.p_vrp.y >= 0) ? 0 : 360);
    psp->polar.psi=(acos(psp->polar.vrel.z/
			 (sqrt(psp->polar.vrel.x*psp->polar.vrel.x+
			       psp->polar.vrel.y*psp->polar.vrel.y+
			       psp->polar.vrel.z*psp->polar.vrel.z))))*DEG;
    psp->polar.chi=((psp->polar.vrel.y >= 0) ? 1 : -1)*
	(acos(psp->polar.vrel.x/
	      (sqrt(psp->polar.vrel.x*psp->polar.vrel.x+
		    psp->polar.vrel.y*psp->polar.vrel.y))))*DEG
			+((psp->polar.vrel.y >= 0) ? 0 : 360);

    /*
     * FIXED ``AXIAL'' Co-ordinates (xa,ya,za)=(beam,horiz,vert)
     * Transformation from Polar Co-ord frame (x,y,z) ---> (za,xa,ya)
     */
    
    psp->axial.p_li.x=psp->polar.p_li.z;
    psp->axial.p_li.y=psp->polar.p_li.x;
    psp->axial.p_li.z=psp->polar.p_li.y;
    p_li_azy=sqrt(psp->axial.p_li.y*psp->axial.p_li.y + 
		  psp->axial.p_li.z*psp->axial.p_li.z);   
    tha_li = ((psp->axial.p_li.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_li.y/p_li_azy))*DEG
	    +((psp->axial.p_li.z >= 0) ? 0 : 360);

    psp->axial.p_hi.x=psp->polar.p_hi.z;
    psp->axial.p_hi.y=psp->polar.p_hi.x;
    psp->axial.p_hi.z=psp->polar.p_hi.y;
    p_hi_azy=sqrt(psp->axial.p_hi.y*psp->axial.p_hi.y + 
		  psp->axial.p_hi.z*psp->axial.p_hi.z);    
    tha_hi = ((psp->axial.p_hi.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_hi.y/p_hi_azy))*DEG
	    +((psp->axial.p_hi.z >= 0) ? 0 : 360);
    
    psp->axial.p_vrp.x=psp->polar.p_vrp.z;
    psp->axial.p_vrp.y=psp->polar.p_vrp.x;
    psp->axial.p_vrp.z=psp->polar.p_vrp.y;

    p_vrp_azy=sqrt(psp->axial.p_vrp.y*psp->axial.p_vrp.y + 
		  psp->axial.p_vrp.z*psp->axial.p_vrp.z);
    dths = ((psp->axial.p_vrp.z >= 0) ? 1 : -1)*
	(acos(psp->axial.p_vrp.y/p_vrp_azy))*DEG
	    +((psp->axial.p_vrp.z >= 0) ? 0 : 360); 
    
    psp->axial.vrel.x=psp->axial.p_li.x/mli-psp->axial.p_hi.x/mhi;
    psp->axial.vrel.y=psp->axial.p_li.y/mli-psp->axial.p_hi.y/mhi;
    psp->axial.vrel.z=psp->axial.p_li.z/mli-psp->axial.p_hi.z/mhi;
    
    psp->axial.ths=(acos(psp->axial.p_vrp.z/pcm))*DEG;

    if((psp->axial.phis=(atan2(psp->axial.p_vrp.y,psp->axial.p_vrp.x))*DEG)<0)
	psp->axial.phis+=360;
    
    psp->axial.psi=(acos(psp->axial.vrel.z/
			 (sqrt(psp->axial.vrel.x*psp->axial.vrel.x+
			       psp->axial.vrel.y*psp->axial.vrel.y+
			       psp->axial.vrel.z*psp->axial.vrel.z))))*DEG;
    if((psp->axial.chi=(atan2(psp->axial.vrel.y,psp->axial.vrel.x))*DEG)<0)
	psp->axial.chi+=360;
    
    /*
     * FLOATING ``BASEL'' Co-ords (xb,yb,zb) => (ths_b=90,phis_b,psi_b,chi_b)
     * ebye Rotn about xa to define trajectory in the reac plane (xb,yb)
     */
    
    psp->basel.p_li.x=psp->axial.p_li.x;
    psp->basel.p_li.y=p_li_azy*(cos((tha_li-dths)*RAD));
    psp->basel.p_li.z=p_li_azy*(sin((tha_li-dths)*RAD));
    
    psp->basel.p_hi.x=psp->axial.p_hi.x;
    psp->basel.p_hi.y=p_hi_azy*(cos((tha_hi-dths)*RAD));
    psp->basel.p_hi.z=p_hi_azy*(sin((tha_hi-dths)*RAD));
    
    psp->basel.p_vrp.x=psp->axial.p_vrp.x;
    psp->basel.p_vrp.y=p_vrp_azy;
    psp->basel.p_vrp.z=0;
    
    psp->basel.vrel.x=psp->basel.p_li.x/mli-psp->basel.p_hi.x/mhi;
    psp->basel.vrel.y=psp->basel.p_li.y/mli-psp->basel.p_hi.y/mhi;
    psp->basel.vrel.z=psp->basel.p_li.z/mli-psp->basel.p_hi.z/mhi;
    
    psp->basel.ths=(acos(psp->basel.p_vrp.z/pcm))*DEG;
    psp->basel.phis=(acos(psp->basel.p_vrp.x/
			  (sqrt(psp->basel.p_vrp.x*psp->basel.p_vrp.x+
				psp->basel.p_vrp.y*psp->basel.p_vrp.y))))*DEG; 
    psp->basel.psi=(acos(psp->basel.vrel.z/
			 (sqrt(psp->basel.vrel.x*psp->basel.vrel.x+
			       psp->basel.vrel.y*psp->basel.vrel.y+
			       psp->basel.vrel.z*psp->basel.vrel.z))))*DEG;
    
    if((psp->basel.chi=(atan2(psp->basel.vrel.y,psp->basel.vrel.x))*DEG)<0)
	psp->basel.chi+=360;
    
    return OK;

}
