#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include "dedx.h"

/*
 * When passed, medium is one of:
 * 	2 letter standard compound id, for example, pp
 * 	1 or 2 letter element name, for example, c
 * 	FO:chemical_formula, for example, C3H8
 * 	OT:list_of_elements, for example, 3,6,12,8,1,1
 *
 * Medium may be suffixed with:
 *	/density
 *	@pressure
 *	!gaseous
 *	!solid
 *
 * Internally, medium is stored as the flags needed to generate that medium
 * with dedx.
 *
 * Ion is one of:
 *	Ion name, for example, 12C
 *	mass,charge, for example, 12,6
 */

static struct dedx_list {
    double thickness;
    double estep;
    double tolerance;
    char *medium;
    char *ion;
    int flags;
    int cookie;
    struct energy_list *energies;
    struct dedx_list *next;
} *dedx_head = NULL;

struct energy_list {
    double emin;
    double emax;
    int order;
    double *coeffs;		/* stored with highest coeff first */
    struct energy_list *next;
};

#define CACHE_FILE "sunsort_dedx.cache"
#define RESULT_FILE ".sunsort_dedx"
#define FILE_FORMAT 1

#define DEFAULT_EMAX 200.0
#define DEFAULT_ESTEP 0.1
#define DEFAULT_TOLERANCE 0.01 

static double emax = DEFAULT_EMAX;
static double estep = DEFAULT_ESTEP;
static double tolerance = DEFAULT_TOLERANCE;
static int next_cookie = 0;

void dedx_emax(double e)
{
    if (e <= 0)
	emax = DEFAULT_EMAX;
    else
	emax = e;
}

void dedx_estep(double e)
{
    if (e <= 0)
	estep = DEFAULT_ESTEP;
    else
	estep = e;
}

void dedx_tolerance(double t)
{
    if (t <= 0)
	tolerance = DEFAULT_TOLERANCE;
    else
	tolerance = t;
}

static void dedx_reload()
{
    dedx_t dp, last;
    struct energy_list *ep, *en;
    FILE *fp;
    int format, flags, l;
    char buffer[256], *medium = NULL, *ion = NULL, *p, ch;
    double est, thick, tol, c[6];

    for(dp = dedx_head; dp != NULL; dp = dp->next)
    {
	for(ep = dp->energies; ep != NULL; ep = en)
	{
	    en = ep->next;
	    free(ep->coeffs);
	    free(ep);
	}
	dp->energies = NULL;
    }

    if ((fp = fopen(CACHE_FILE, "r")) == NULL)
	return;

    while(fgets(buffer, 256, fp) != NULL)
    {
	switch(*buffer)
	{
	case '#':
	    sscanf(buffer+2, "%d", &format);
	    dp = NULL;
	    break;
	case '*':
	    if (format == 1)
	    {
		if (medium != NULL)
		    free(medium);
		for(p = buffer+2; isprint(*p); p++)
		    ;
		l = p - buffer - 2;
		if ((medium = malloc(l+1)) == NULL)
		{
		    perror("dedx_reload, medium");
		    fclose(fp);
		    return;
		}
		memcpy(medium, buffer+2, l);
		medium[l] = '\0';
	    }
	    break;
	case '+':
	    if (format == 1)
	    {
		if (ion != NULL)
		    free(ion);
		for(p = buffer+2; *p != ' ' && *p != '\0'; p++)
		    ;
		l = p - buffer - 2;
		if ((ion = malloc(l)) == NULL)
		{
		    perror("dedx_reload, ion");
		    fclose(fp);
		    return;
		}
		memcpy(ion, buffer+2, l);
		ion[l] = '\0';
		if (sscanf(buffer+2+l, "%lf %lf %c %lf", &est, &thick, &ch,
			   &tol) != 4)
		{
		    fprintf(stderr, "dedx_reload: Unable to parse %s", buffer);
		    fclose(fp);
		    return;
		}
		if (ch == 'A')
		    flags = DEDX_AFTER;
		else
		    flags = DEDX_BEFORE;
	    }
	    break;
	default:
	    if (format == 1)
	    {
		if (dp == NULL)
		{
		    for(dp = dedx_head; dp != NULL; dp = dp->next)
		    {
			if (!strcmp(dp->medium, medium) &&
			    !strcasecmp(dp->ion, ion) && dp->estep == est &&
			    dp->thickness == thick && dp->flags == flags &&
			    dp->tolerance == tolerance)
			    break;
			last = dp;
		    }
		    if (dp == NULL)
		    {
			if ((dp = (dedx_t)
			     malloc(sizeof(struct dedx_list))) == NULL)
			{
			    perror("dedx_reload, new");
			    fclose(fp);
			    return;
			}
			if ((dp->medium = strdup(medium)) == NULL)
			{
			    perror("dedx_reload, new");
			    free(dp);
			    fclose(fp);
			    return;
			}
			if ((dp->ion = strdup(ion)) == NULL)
			{
			    perror("dedx_reload, new");
				free(dp->medium);
				free(dp);
				fclose(fp);
				return;
			}
			dp->estep = est;
			dp->thickness = thick;
			dp->flags = flags;
			dp->tolerance = tolerance;
			dp->energies = NULL;
			dp->next = NULL;
			dp->cookie = next_cookie++;
			if (dedx_head == NULL)
			    dedx_head = dp;
			else
			    last->next = dp;
		    }
		}
		if ((ep = (struct energy_list *)
		     malloc(sizeof(struct energy_list))) == NULL)
		{
		    perror("dedx_reload, energy");
		    fclose(fp);
		    return;
		}
		ep->order = sscanf(buffer, "%lf %lf %lf %lf %lf %lf %lf %lf",
				   &ep->emin, &ep->emax, c, c+1, c+2, c+3,
				   c+4, c+5) - 2;
		if (ep->order < 1)
		{
		    fprintf(stderr, "Malformed energy list %s\n", buffer);
		    free(ep);
		    fclose(fp);
			return;
		}
		if ((ep->coeffs = (double *)
		     malloc(ep->order*sizeof(double))) == NULL)
		{
		    perror("dedx_reload, energy");
		    free(ep);
		    fclose(fp);
		    return;
		}
		for(l=ep->order; l--; )
		    ep->coeffs[l] = c[ep->order-l-1];
		ep->next = dp->energies;
		dp->energies = ep;
	    }
	    break;
	}
    }
    if (medium != NULL)
	free(medium);
    if (ion != NULL)
	free(ion);
    fclose(fp);
}

static void dedx_calc(dedx_t dp, double emax)
{
    char *buffer, *p;
    FILE *ip, *op;
    int i, mass, charge;

    if ((buffer = malloc(strlen(dp->medium)+strlen(dp->ion)+256)) == NULL)
    {
	perror("dedx_calc");
	return;
    }
    sprintf(buffer, "dedx -medium %s -emin 0 -emax %f -estep %f -thickness %f "
	    "-%s -tolerance %f", dp->medium, emax, dp->estep, dp->thickness,
	    (dp->flags & DEDX_AFTER) ? "after" : "before", dp->tolerance);
    if ((p = strchr(dp->ion, ',')) == NULL)
	sprintf(buffer+strlen(buffer), " -ion %s", dp->ion);
    else
    {
	sscanf(dp->ion, "%d", &mass);
	sscanf(p+1, "%d", &charge);
	sprintf(buffer+strlen(buffer), " -mass %d -charge %d", mass, charge);
    }
    fprintf(stderr, "%s\n", buffer);
    if (system(buffer) < 0)
    {
	perror("dedx_calc");
	return;
    }
    free(buffer);
    
    if ((ip = fopen(RESULT_FILE, "r")) == NULL)
    {
	fprintf(stderr, "dedx_calc: Dedx did not return its results. Please "
		"check this window for\nerror messages, and check your "
		"version of dedx is up to date.\n");
	return;
    }
    if ((op = fopen(CACHE_FILE, "a")) == NULL)
    {
	perror("dedx_calc, " CACHE_FILE);
	fclose(ip);
	return;
    }
    fprintf(op, "# %d\n", FILE_FORMAT);
    fprintf(op, "* %s\n", dp->medium);
    fprintf(op, "+ %s %f %f %c %f\n", dp->ion, dp->estep, dp->thickness,
	    (dp->flags & DEDX_AFTER) ? 'A' : 'B', dp->tolerance);
    while((i = getc(ip)) != EOF)
	putc(i, op);
    fclose(op);
    fclose(ip);
    remove(RESULT_FILE);

    dedx_reload();
}

double dedx(dedx_t dp, double e)
{
    struct energy_list *ep = dp->energies, *sp = NULL;
    double de, *c;
    int i;

    while(ep != NULL && (ep->emin > e || ep->emax < e))
    {
	if (e < ep->emin && (sp == NULL || ep->emin < sp->emin))
	    sp = ep;
	ep = ep->next;
    }

    if (ep == NULL)
    {
	if (sp != NULL)
	{
	    if (dp->flags == DEDX_BEFORE)
		return e;
	    e = sp->emin;
	    ep = sp;
	}
	else
	{
	    dedx_calc(dp, (e > emax) ? e*2 : emax);
	    ep = dp->energies;
	    sp = NULL;
	    while(ep != NULL && (ep->emin > e || ep->emax < e))
	    {
		if (e < ep->emin && (sp == NULL || ep->emin < sp->emin))
		    sp = ep;
		ep = ep->next;
	    }
	    if (ep == NULL)
	    {
		if (sp != NULL)
		{
		    if (dp->flags == DEDX_BEFORE)
			return e;
		    e = sp->emin;
		    ep = sp;
		}
		else
		{
		    fprintf(stderr, "dedx: Failed to calculate de/dx\n");
		    return -1.0;
		}
	    }
	}
    }

    c = ep->coeffs;
    de = *c++;
    for(i = ep->order; --i; )
	de = de*e + *c++;

    return de;
}

dedx_t dedx_setup(char *medium, char *ion, int flags, double thick)
{
    dedx_t dp, sp, last;
    char *p, *q, *r;
    int i;

    if (dedx_head == NULL)
	dedx_reload();

    if (*medium == '\0' || *ion == '\0')
    {
	fprintf(stderr, "dedx_setup: invalid parameters.\n");
	return NULL;
    }

    if ((dp = (dedx_t) malloc(sizeof(struct dedx_list))) == NULL)
    {
	perror("dedx_setup");
	return NULL;
    }
    
    i = 50; /* safety margin */
    for(p = medium; *p != '\0'; p++)
	switch(*p)
	{
	case ':':
	    i += 20;
	    break;
	case '@':
	    i += 11;
	    break;
	case '/':
	    i += 10;
	    break;
	case '!':
	    i += 2;
	    break;
	default:
	    i++;
	    break;
	}

    if ((dp->medium = malloc(i)) == NULL)
    {
	perror("dedx_setup");
	free(dp);
	return NULL;
    }

    p = medium;
    q = dp->medium;

    while(isalpha(*p))
	*q++ = toupper(*p++);

    if (dp->medium[0] == 'F' && dp->medium[1] == 'O' && *p == ':')
    {
	strcpy(q, " -formula ");
	p++;
	q += 10;
	while(*p != '@' && *p != '/' && *p != '!' && *p != '\0')
	    *q++ = *p++;
    }
    else if (dp->medium[0] == 'O' && dp->medium[1] == 'T' && *p == ':')
    {
	p++;
	i = 0;
	for(r = p; *r != '@' && *r != '/' && *r != '!' && *r != '\0'; r++)
	    if (*r == ',')
		i++;
	sprintf(q, " -elementlist %d ", (i+1)/3);
	while(*q != '\0')
	    q++;
	while(*p != '@' && *p != '/' && *p != '!' && *p != '\0')
	{
	    if (*p == ',')
		*q++ = ' ';
	    else
		*q++ = *p;
	    p++;
	}
    }

    while(*p != '\0')
    {
	switch(*p)
	{ 
	case '@':
	    strcpy(q, " -pressure ");
	    q += 11;
	    break;
	case '/':
	    strcpy(q, " -density ");
	    q += 10;
	    break;
	case '!':
	    *q++ = ' ';
	    *q++ = '-';
	    break;
	default:
	    *q++ = *p;
	    break;
	}
	p++;
    }
    *q++ = '\0';

    if ((p = realloc(dp->medium, strlen(dp->medium)+1)) != NULL)
	dp->medium = p;

    if ((dp->ion = strdup(ion)) == NULL)
    {
	perror("dedx_setup");
	free(dp->medium);
	free(dp);
	return NULL;
    }

    dp->thickness = thick;
    dp->estep = estep;
    dp->tolerance = tolerance;
    dp->flags = flags;

    for(sp = dedx_head; sp != NULL; sp = sp->next)
    {
	if (!strcmp(dp->medium, sp->medium) &&
	    !strcasecmp(dp->ion, sp->ion) && dp->estep == sp->estep &&
	    dp->thickness == sp->thickness && dp->flags == sp->flags &&
	    dp->tolerance == sp->tolerance)
	    break;
	last = sp;
    }

    if (sp != NULL)
    {
	free(dp);
	return sp;
    }

    dp->cookie = next_cookie++;
    dp->energies = NULL;
    dp->next = NULL;

    if (dedx_head == NULL)
	dedx_head = dp;
    else
	last->next = dp;

    return dp;
}

int dedx_setup_(char *medium, char *ion, int *flags, double *thick)
{
    dedx_t dp;
    
    dp = dedx_setup(medium, ion, *flags, *thick);
    if (dp == NULL)
	return -1;

    return dp->cookie;
}

double dedx_(int cookie, double *e)
{
    dedx_t dp;
    
    for(dp = dedx_head; dp->cookie != cookie; dp=dp->next)
	;

    return dedx(dp, *e);
}

void dedx_emax_(double *e)
{
    dedx_emax(*e);
}

void dedx_estep_(double *e)
{
    dedx_estep(*e);
}

void dedx_tolerance_(double *t)
{
    dedx_tolerance(*t);
}
