/*
 * ''nrp''      N Fold Resonant Particles                  Steve Chappell
 *
 * General Subroutine to Generate resonant particles 
 * from N breakup products and add the particles to the
 * the ev array.

 * Notes

 * Includes subroutine ''ncr'' to calculate r combinations from n

 * Reconstruct Pairs from two singles
 * Reconstruct Triples from Pairs plus singles
 * Reconstruct Quads from Triples plus singles
 * etc etc
 * Result is ``heavy ion'' and ``light ion''  point back up the tree 
 * to the single particle numbers and DO NOT correspond to the 
 * sequential breakup into HI and LI.

 * In the event array.... If N=nhits  nCr combinations of each type
 * Singles  1              --> N
 * Doubles  N +1           --> N +1 +NC2-1
 * Triples  N +1 NC2-1 +1  --> N +1 +NC2-1 +1 +NC3-1
 * Quads

 * Updates
 *   24th Jun 97: Translation to C
 *   17th Jul 97: Streamline code to handle any value of N
 *   9th Jun 98:  Streamline code to generate up to the Nfold rp.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "evsubs.h"

static void combine(int previ, particle **vrp, particle *lastvrp, int nfold,
		    struct evhead *evh)
{
    int i;
    
    for(i=previ+1; i <= evh->n[1]; i++)
    {
	if (lastvrp != NULL){
	    rp(*vrp, lastvrp, evh->ev+i);
	    /*fprintf(stderr,"(%d %d)=%d\n",lastvrp,i,*vrp);*/
	}
	if(ev_fold(*vrp) < nfold)
	    combine(i, vrp+1, *vrp, nfold, evh);
	(*vrp)++;
    }
}

int nrp(struct evhead *evh, int fold)
{
    particle **vrp;
    int i;
    
    /*make sure we don't try and build an Nfold combination 
      for too few singles*/
    if(fold > evh->n[1])
	fold = evh->n[1];
    
    if(fold < 2)
	return OK;
    
    if ((vrp = (particle **) malloc(fold*sizeof(particle *))) == NULL)
    {
	fprintf(stderr, "nrp: Out of memory.\n");
	return ABORT;
    }

/*
 * Put Number of different Nfold combinations
 * Plus Pointers to ``start'' and ``Next'' locations 
 * of Nfold particles in the Event Header
 */
    
    vrp[0]=evh->ev+evh->p[1];
    
    for(i=2; i <= fold; i++)
    {
	evh->n[i] = ncr(evh->n[1],i);
	evh->p[i] = evh->p[i-1] + evh->n[i-1];
	evh->n[0] = evh->n[0] + evh->n[i];
	vrp[i-1] = evh->ev+evh->p[i];
	if(evh->n[0] > evh->numparticles)
	{
	    fprintf(stderr,"nrp: Not building rp combinations\n"
		    "...Cannot fit nrp(%d) for %d particles in ev array.\n" 
		    "Declared size =%d\n" ,fold, evh->n[1], evh->numparticles);
	    free(vrp);
	    return ABORT;
	}
    }
    
    /* Build the combinations of virtual fold particles*/
    combine(0, vrp, 0, fold, evh);
    free(vrp);
    
    return OK;
}


/*
 *''ncr''  Calculate r combinations out of n               Steve Chappell
 */
int ncr(int n_ncr, int r_ncr)
{
    int i_ncr;
    int value=1;
    if(n_ncr < 0 || r_ncr < 0 || n_ncr < r_ncr){
        fprintf(stderr,"ncr: Error! n=%d r=%d\n",n_ncr,r_ncr);
	return OK;
    }
    for(i_ncr=1; i_ncr<=r_ncr; i_ncr++)
	value=(n_ncr-i_ncr+1)*value/i_ncr;
    /*fprintf(stderr,"ncr: n=%d r=%d = %d\n",n_ncr,r_ncr,value);*/
    return value;
}

int nrp_(int *fold)
{
    return nrp(&evhead, *fold);
}

