/*
 * ''reac''      Reaction definition                           Steve Chappell
 *
 *   General Subroutine to setup default reaction parameters
 *   from file
 
 * Updates
 *   26th Jun 97
 *
 */

#include <stdio.h>
#include <math.h>
#include "evsubs.h"

struct reac reac;

/*
 * Reaction definition routine
 */

int reac_init(int opt_flag)
{
    FILE *fp;
    char buffer[SUBS_BUFSIZE];
    double m0,m1,m2,e1,edisp;
    /* Define this once to save space and typing */
    
    /*default settings*/
    if(opt_flag == REAC_SIM){
	reac.edisp=1.0;
	reac.e1=reac.edisp*sim.e1;
	reac.mass[0]=0.0;
	reac.mass[1]=ev_m(evsim + 1);
	reac.mass[2]=ev_m(evsim + 2);
    }
    else if(opt_flag == REAC_SORT){
	reac.edisp=1.0;
	reac.e1=reac.edisp*65.0;
	reac.mass[0]=4.0;
	reac.mass[1]=12.0;
	reac.mass[2]=12.0;
    }
    else{
	fprintf(stderr,"reac: Error! unknown option %d\n",opt_flag);
	return ABORT;
    }
    
    if ((fp = fopen(CALIB_DIR "reac.cal", "r")) != NULL)
    {
	fprintf(stderr, "reac: Reading file \"%s\".\n",CALIB_DIR "reac.cal");
	MYGETLINE(buffer, SUBS_BUFSIZE, fp,
		  "reac: End of file reading e1 e1disp.\n", 1)
	if (sscanf(buffer, " %lf %lf", &e1, &edisp) != 2)
	{
	    fprintf(stderr, "reac:Expected 'e1 edisp', got \"%s\".\n", buffer);
	    return ABORT;
	}
	reac.e1= e1*edisp;
	reac.edisp= edisp;
	
	MYGETLINE(buffer, SUBS_BUFSIZE, fp,
		  "reac: End of file reading masses.\n", 1)
	if (sscanf(buffer, " %lf %lf", &m1, &m2) != 2)
	{
	    fprintf(stderr, "reac:Expected 'mass1 mass2'"
		    "got \"%s\".\n", buffer);
	    return ABORT;
	}
	reac.mass[1]= m1;
	reac.mass[2]= m2;
	
	MYGETLINE(buffer, SUBS_BUFSIZE, fp,
		  "reac: End of file reading m0.\n", 1)
	if (sscanf(buffer, " %lf", &m0) != 1)
	{
	    fprintf(stderr, "reac:Expected 'm0', got \"%s\".\n", buffer);
	    return ABORT;
	}
	reac.mass[0]= m0;

	fclose(fp);
    }
    else
    {
	fprintf(stderr, "reac: No file \"%s\", using default settings.\n",
		CALIB_DIR "reac.cal");
    }
    
    /*additional calculated quantities*/
    reac.p1=sqrt(reac.e1*2.0*reac.mass[1]);
    reac.vcm=reac.p1/(reac.mass[1]+reac.mass[2]);

    if(opt_flag == REAC_SIM)
	fprintf(stderr, "reac: Pseudodata Simulation for...");
    
    fprintf(stderr,"\nreac: Default Sort Reaction:\n"
	    "%5.1f[%7.2f MeV] ->%5.1f ==> ...",
	    reac.mass[1],reac.e1/reac.edisp,reac.mass[2]);
    fprintf(stderr,"Hitmass=%5.1f\n",reac.mass[0]);
    fprintf(stderr,"  Edisp = %7.2f Channs/MeV\n\n",reac.edisp);
    
    return OK;
}


/*fortran compatability*/

int reac_init_(int i)
{
    return(reac_init(i));
}

double reac_edisp(void){ return(reac.edisp); }
void reacs_edisp(double v){ reac.edisp = v; }
double reac_edisp_(void){ return(reac.edisp); }
void reacs_edisp_(double *v){ reac.edisp = *v; }

double reac_e1(void){ return(reac.e1); }
void reacs_e1(double v){ reac.e1 = v; }
double reac_e1_(void){ return(reac.e1); }
void reacs_e1_(double *v){ reac.e1 = *v; }

double reac_p1(void){ return(reac.p1); }
void reacs_p1(double v){ reac.p1 = v; }
double reac_p1_(void){ return(reac.p1); }
void reacs_p1_(double *v){ reac.p1 = *v; }

double reac_mass(int i){ return(reac.mass[i]); }
void reacs_mass(int i, double v){ reac.mass[i] = v; }
double reac_mass_(int *i){ return(reac.mass[(*i)]); }
void reacs_mass_(int *i, double *v){ reac.mass[(*i)] = *v; }

double reac_vcm(void){ return(reac.vcm); }
void reacs_vcm(double v){ reac.vcm = v; }
double reac_vcm_(void){ return(reac.vcm); }
void reacs_vcm_(double *v){ reac.vcm = *v; }


