/* 
 * ''scanga''   Event Simulation Routines               Steve Chappell

 *   ``RPS/Monte Carlo'' routines to generate pseudo data
 *   for heavy ion breakup reactions  using Charissa detection systems

 *   Uses Lab Fixed Co-ords (X,Y,Z) = (Hor,Vert,Beam)

 * Updates
 *   12th Dec97 :Linked to SunSort!
 *   17th Dec97 :Convert to Lab fixed co-ord convention. 
 *              :Angles converted to positive range 0->360.
 *              :Code added to generate isotropic event distribution.
 *   11th Jan98 :Gaussian random number generator smears detected energies
 *   7th May98  :Writes hits to the event array
 *   1st Jun98  :Basel correlation function included
 *   19th Jun98 :2nd partial wave in basel corr fn added.
 *   9th Feb99  :Punchthru calculation
 */

#include <stdio.h> 
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <sys/time.h>
#include "../evsubs.h"

/* This is now local to this routine, set whatever granularity you like */
#define HITBINSIZE 2

struct sim sim;
particle *evsim = NULL;

static int hitsave[SIM_NPMAX];
static int nhits;
static double vcm,vrel[SIM_NPMAX];
static int rorder[SIM_NPMAX];
static void grorder(int);

/*
 * Initialise Scanga
 */
int scanga_init(char *filename)
{
    FILE *fp;
    int i,j,k;
    double th,phi,tmp;
    struct timeval tp;
    char *star1,*star2;
    
    fprintf(stderr,"\nScanga: Initialising...\n");

    /*seed random number generator*/
    gettimeofday(&tp, NULL);
    srand48(tp.tv_sec + tp.tv_usec);

    if(evsim_init(SIM_NPMAX)) return ABORT;/*initialise sim event structure*/
    if(ev_init(&evhead, &reac, NDET_MAX, N1_MAX, NP_MAX))
	return ABORT;      /*initialise event structure*/

    /*zero kinematic arrays*/
    /*zero arrays*/
    for(i=1;i<SIM_NPMAX;i++){
	sim.fstate[i] = sim.hit[i] = sim.bkup[i] =0;
	sim.ex[i] = sim.w[i] = 0.0;
    }
    
    if(scanga_input(filename)) return ABORT;   /*Read Input File*/
    if (reac_init(REAC_SIM)) return ABORT;     /*reaction for pseudo data*/

    /*Generate detector coverage*/
    if(sim.det_sim){
	
	/*This code tests for detector hits on the fly*/
	/*The alternative is to always setup the array 
	 *gethitbin(th,phi) by running the code here*/

	inithitbins((double) HITBINSIZE, (double) HITBINSIZE);
	
	if(sim.thabs > 0) fprintf(stderr,"Scanga: Absorber shields up to "
				  "Theta = %6.2f deg\n",sim.thabs);
	
	/*Check detector coverage if requested*/
	if(sim.cov){
	    if((fp = fopen("sim_cov.out","w"))!= NULL){
		fprintf(stderr, "Scanga: Writing complete th,phi coverage\n"
			" to file \"%s\".\n","sim_cov.out");
	    }
	    for(i=1; i<sim.ndet+1; i++){
		fprintf(stderr, "det %d: ",i);
		switch(sim.det_type[i]){
		case DTYPE_S:
		    fprintf(stderr,"(S)trip Array detector\n");
		    break;
		case DTYPE_a:
		    fprintf(stderr,"(a)ncillary detector\n");
		    break;
		default:
		    fprintf(stderr,"Scanga: Error initialising"
			    " detector coverage\n");
		    return ABORT;
		}
		fprintf(fp,"\n");
		for(th=1/(2.0*HITBINSIZE); th<180; th += 1.0/HITBINSIZE){
		    if(th < sim.thabs)
			continue;
		    for(phi=1/(2.0*HITBINSIZE); 
			phi<360; phi += 1.0/HITBINSIZE)
		    {
			if(*gethitbin(th, phi) = det_hit(i,th,phi))
			    fprintf(fp, "%f %f\n", th,phi);
		    }
		}
	    }
	    fclose(fp);
	}
    }

    /* If we've finished with this array, we can ditch it */
    freehitbins();

    /*Recap on final state particles and deduced reaction*/

    fprintf(stderr,"\nScanga: Deduced Reaction for Simulation:\n");
    
    if(sim.bkup[3])
	star1 = "*";
    else 
	star1 = " ";
    if(sim.bkup[4])
	star2 = "*";
    else
	star2 = " ";
    
    fprintf(stderr,"  1  [%-5s (%6.2f MeV)] + 2 [%-5s]"
	    " -> 3%s + 4%s : Q1=%7.3f\n",
	    nucleusname_az((int) (ev_m(evsim+1)+0.5), ev_z(evsim+1)),
	    sim.e1,
	    nucleusname_az((int) (ev_m(evsim+2)+0.5), ev_z(evsim+2)),
	    star1,star2,sim.q1);

    for(i=3; i<sim.np+3; i++){
	
	if(sim.det_sim){
	    if(sim.coinc[i]== 1 && ev_m(evsim + i) <= 0.000001){
		fprintf(stderr,"Scanga: Canny detect massless particles!\n");
		return ABORT;
	    }
	    if(sim.coinc[i]== 1 &&
	       (sim.bkup[i] == 1 ||
		sim.coinc[ev_hi(evsim + i) - evsim] == 1 
		|| sim.coinc[ev_li(evsim + i) - evsim] == 1)){
		fprintf(stderr,
			"Scanga: Canny detect particle %d AND its breakup!\n",
			i);
		return ABORT;
	    }
	}
	
	
	if(sim.bkup[i]){
	    star1 = "*";
	    sim.fstate[i] = 0;
	}
	else{
	    star1 = " ";
	    sim.fstate[i] = 1;
	}
	fprintf(stderr,"%3d%s [%-5s (%6.2f +/- %6.2f)]",
		i,star1,
		nucleusname_az((int) (ev_m(evsim+i)+0.5), ev_z(evsim+i)),
		sim.ex[i],sim.w[i]);	
	if(sim.bkup[i]){
	    if(sim.bkup[ev_hi(evsim + i) - evsim])
		star1 = "*";
	    else
		star1 = " ";
	    if(sim.bkup[ev_li(evsim + i) - evsim])
		star2 = "*";
	    else
		star2 = " ";
	    fprintf(stderr," -> %3d%s + %3d%s :qb = %7.3f",
		    ev_hi(evsim+i) - evsim, star1,
		    ev_li(evsim+i) - evsim, star2, sim.qb[i]);
	}
	fprintf(stderr,"\n");
	
    }

  
    fprintf(stderr,"\n=>Final State Particle detection...\n\n");
    nhits = 0;

    if(sim.det_sim)
	fprintf(stderr,"Low E limit (MeV) and hitpats for Dets 1-%d\n",
		sim.ndet);
    else if(sim.det_t_sim)
	fprintf(stderr,"Low E limit (MeV) only\n");
    else
	fprintf(stderr,"No E threshold simulation\n");
	
    /*Low Energy Thresholds and hit patterns*/
    for(i=3;i<sim.np+3;i++){

	if(sim.fstate[i] < sim.coinc[i]){
	    fprintf(stderr,"\nScanga: Error! fstate < coinc requirement\n");
	    return ABORT;
	}
	
	/*Nb: Want to ``see'' all final state particles when 
	  no detector simulation required*/
	if(sim.fstate[i]){
	    sim.hit[i] = 1;
	    nhits++;
	    hitsave[nhits] = i; /*remember simulated particle index*/
	    evs_det(evsim + i, 0);
	}
	

	if(sim.fstate[i]){
	    fprintf(stderr,"%3d",i);
	    if(sim.coinc[i])
		fprintf(stderr,".");
	    else
		fprintf(stderr," ");
	    if(sim.det_pthru)
		fprintf(stderr," [%-5s] :",
			nucleusname_az((int) (ev_m(evsim+i)+0.5),
				       ev_z(evsim+i)));
	    if(sim.det_sim || sim.det_t_sim){
		for(k=1;k<sim.ndet+1;k++){
		    fprintf(stderr,"%5.1f ",sim.hitpat[k][i]);
		}
	    }
	    fprintf(stderr,"\n");
	}

    }  
    fprintf(stderr,"[required particle detection marked by . ]\n\n");

    if(sim.det_pthru){
	fprintf(stderr,"Upper Range limit (microns) for Dets 1-%d\n",sim.ndet);
	for(i=1;i<sim.ndet+1;i++){
	    fprintf(stderr,"%5.0f ",sim.det_thickness[i]);
	}
	    fprintf(stderr,"\n\n");
    }
    else
	fprintf(stderr,"No Range limit simulation\n\n");
    
    /*Mcarlo Set-up*/
    if(sim.effcalc == 0){
	sim.effmin=0.0;
	sim.effmax=0.0;
	sim.effstep=1.0;
	fprintf(stderr,"Scanga: Generating Pseudo Data \n");
    }
    else
	fprintf(stderr,"Scanga: Generating Pseudo Data "
		"and efficiencies\n");

    if(sim.random)
	fprintf(stderr,"        with ev array in random order...\n");
    else
	fprintf(stderr,"        with ev array in evsim order...\n");

    /*End of Set-up*/
    return OK;
}

/*
 * Initialise kinematics and coordinates
 */
int scanga_kin_init(void)
{
    double vcm,erel,qbtot;
    int i;
    /*first stage of reaction*/
    sim.bkup[2] = 1;
    evs_hi(evsim + 2, evsim + 3);
    evs_li(evsim + 2, evsim + 4);
    sim.qb[2] = sim.q1;
    evs_q(evsim + 2, sim.ex[2] =
	  ev_m(evsim + 2)*sim.e1/(ev_m(evsim + 1)+ev_m(evsim + 2)));
    vcm = (sqrt(2.0*ev_m(evsim + 1)*sim.e1))/(ev_m(evsim + 1)+ev_m(evsim + 2));
    evs_v_xyz(evsim + 2, 0.0, 0.0, vcm);

    qbtot = 0.0;
    for(i=2;i<sim.np+3;i++){
	qbtot = qbtot + sim.qb[i];
    }
    erel=ev_q(evsim + 2)-sim.ex[3]-sim.ex[4] + qbtot;
    if(erel <= 0.0){
	fprintf(stderr,"Scanga: Warning -ve energy for final state!\n");
    }

    return OK;
}


/*
 * Scanga Event by event Simulation
 */   
int scanga_event(void)
{
    int i,j,k,n,hi,li;
    double vrelx[SIM_NPMAX],vrely[SIM_NPMAX],vrelz[SIM_NPMAX];
    double ths,phs,arg,f,phis,chi[2],psi[2],d,dp,dths, x ,y, z;
    double vcm, tmp, tmp_norm, norm_d, norm_dp;
    double erel, vreltot, mtot;
    struct xy detpos;
    particle *evi, *evhi, *evli, *evk;
   
    ev_reset(&evhead);              /*Reset event structure*/
    
    for(i=3;i<sim.np+3;i++) 
	/*Smear excitations according to widths of states*/
	evs_q(evsim + i, grand(sim.ex[i],sim.w[i]));
    
    for(evi = evsim + 2, i=2;i<sim.np+3;i++, evi++){
	if(sim.bkup[i]){
	    evhi = ev_hi(evi);
	    evli = ev_li(evi);

	    hi = evhi - evsim;
	    li = evli - evsim;
	    
	    if(i == 2)
		mtot=ev_m(evsim + 1)+ev_m(evsim + 2);
	    else
		mtot = ev_m(evi);
	    
	    erel=ev_q(evi)-ev_q(evhi)-ev_q(evli)+sim.qb[i];
	    if(erel <= 0){
		fprintf(stderr,"Scanga:-ve erel for particle %d!\n",i);
		return SKIP;
	    }
	    
	    vreltot=sqrt(2.0*erel*mtot/(ev_m(evhi)*ev_m(evli)));
	    vrel[hi]=vreltot*ev_m(evli)/mtot;
	    vrel[li]=vreltot*ev_m(evhi)/mtot;
	    	    
	    switch(sim.bkup_coords){
		
	    case 'b': /* 
		       *   Breakup using Basel correlation function
		       *   Chappell+Rae PRC 53(1996)2879
		       *   assumes m1+m2 -> m3* + m4*,  m3* -> m5+m6 , 
		       *   [m4*->m7+m8] ...etc
		       *    Nb preserving lab coords (X,Y,Z) = (Hor,Vert,Beam)
		       */
		if(i==2){
		    do{
			do{
			    phis=180.0*drand48();
			} while(drand48() > sin(phis*RAD));
			for(k=0;k<2;k++){
			    do{
				psi[k]=180.0*drand48();
			    } while(drand48() > sin(psi[k]*RAD));
			    chi[k]=360.0*drand48();
			}
			arg=0.0;
			for(k=0;k<sim.b_nl;k++){
			    
			    /*d matrix for m=0 p147 Brink and Satchler*/
			    d=dp=1.0;
			    norm_dp=norm_d=1.0;
			    tmp = cos(0.5*psi[0]*RAD)*sin(0.5*psi[0]*RAD);
			    tmp_norm = cos(0.5*90.0*RAD)*sin(0.5*90.0*RAD);
			    for(n=0;n<sim.b_k[k];n++){
				d *= tmp;
				norm_d *= tmp_norm;
			    }
			    d /= norm_d;

			    tmp = cos(0.5*psi[1]*RAD)*sin(0.5*psi[1]*RAD);
			    for(n=0;n<sim.b_kp[k];n++){
				dp *= tmp;
				norm_dp *= tmp_norm;
			    }
			    dp /= norm_dp;

			    arg += sim.b_al[k]*(sin((sim.b_l[k]+0.5)*phis*RAD +
						    sim.b_k[k]*chi[0]*RAD + 
						    sim.b_kp[k]*chi[1]*RAD + 
						    0.25*M_PI))*d*dp;
			}
			f=(arg*arg)/sin(phis*RAD);
		    } while (drand48() > f);
#if 0
		    evs_pcm_rtpd(evhi, 1.0, phis, 0);
		    evs_pcm_rtpd(evli, 1.0, 180.0-phis, 0);
#endif
		    dths=360.0*drand48(); /*smear over cylindrical geometry*/
		    vrelx[hi]=+vrel[hi]*sin(phis*RAD)*cos(dths*RAD);
		    vrely[hi]=-vrel[hi]*sin(phis*RAD)*sin(dths*RAD);
		    vrelz[hi]=+vrel[hi]*cos(phis*RAD);
		    break;
		}
		else if(i==3 || i==4){
		    j=i-3;
		    evhi=ev_hi(evi);
		    evli=ev_li(evi);
		    hi = evhi - evsim;
		    li = evli - evsim;
		    x = -vrel[hi]*sin(chi[j]*RAD)*sin(psi[j]*RAD);
		    y = -vrel[hi]*cos(psi[j]*RAD);
		    vrelz[hi]= -vrel[hi]*cos(chi[j]*RAD)*sin(psi[j]*RAD);
		    vrelx[hi] = x*cos(dths*RAD) + y*sin(dths*RAD);
		    vrely[hi] = -x*sin(dths*RAD) + y*cos(dths*RAD);
		    break;
		}
		
	    default: /*
		      *    Randomize Over Phase Space using 
		      *    Isotropic breakups only
		      */
		
		if(i==2){ 
		    ths=180.0*drand48();
#if 0
		    evs_pcm_rtpd(evhi, 1.0, ths, 0);
		    evs_pcm_rtpd(evli, 1.0, 180-ths, 0);
#endif
		}
		else
		    do{
			ths=180.0*drand48();
		    }while (drand48() > sin(ths*RAD));
		
		phs=360.0*drand48();
		vrelx[hi]= +vrel[hi]*sin(ths*RAD)*cos(phs*RAD);
		vrely[hi]= +vrel[hi]*sin(ths*RAD)*sin(phs*RAD);
		vrelz[hi]= +vrel[hi]*cos(ths*RAD);
	    }

	    vrelx[li]=-vrelx[hi]*ev_m(evhi)/ev_m(evli);
	    vrely[li]=-vrely[hi]*ev_m(evhi)/ev_m(evli);
	    vrelz[li]=-vrelz[hi]*ev_m(evhi)/ev_m(evli);
	    evs_v_xyz(evhi, vrelx[hi] + ev_v_x(evi), vrely[hi] + ev_v_y(evi), 
		      vrelz[hi] + ev_v_z(evi));
	    evs_p_xyz(evhi, ev_m(evhi)*ev_v_x(evhi), ev_m(evhi)*ev_v_y(evhi),
		      ev_m(evhi)*ev_v_z(evhi));
	    evs_v_xyz(evli, vrelx[li] + ev_v_x(evi), vrely[li] + ev_v_y(evi),
		      vrelz[li] + ev_v_z(evi));
	    evs_p_xyz(evli, ev_m(evli)*ev_v_x(evli), ev_m(evli)*ev_v_y(evli),
		      ev_m(evli)*ev_v_z(evli));
	}
    }
    
    /*==== Simulate Particle Detection============*/
    
    if(sim.det_sim || sim.det_t_sim){
	/*Check which final state particles hit a detector
	  and are at trajectories greater than extent of absorber. 
	  Compare detected hit pattern to coincidence requirement*/
	
	nhits=0;
	for(evi = evsim + i, i=3; i<sim.np+3; i++, evi++){
	    sim.hit[i] = 0;

	    if(sim.fstate[i]){
		double th;

		th = ev_p_td(evi);

		if(sim.det_sim)
		    evs_det(evi, ndet_hit(sim.ndet,th,ev_p_pd(evi)));
		else if(sim.det_t_sim)
		    evs_det(evi, 1);
		
		if(th > sim.thabs && ev_det(evi) &&
		   sim.hitpat[ev_det(evi)][i] >= 0.0 &&
		   ev_e(evi) >= sim.hitpat[ev_det(evi)][i]
		   ){
		    /*Minimum coincidence requirement*/
		    sim.hit[i] = 1;
		    nhits++;
		    hitsave[nhits] = i; /*remember simulated particle index*/
		}
	    }
	    
	    if(sim.hit[i] < sim.coinc[i])
		return 0;          /*detection failed*/
	}
	if(nhits == 0) return 0;   /*detection failed*/
	
    }      /*End detection simulation*/    
    
    
    /*=== Event Good if it made it this far! ===*/
    
    /*load event header*/
    evhead.n[0]=evhead.n[1]=evhead.pmult[0]=nhits;
    
    /*load event array in random sequence if required*/
    if(sim.random) 
	grorder(nhits);
    for(j=1; j<nhits+1; j++){
	
	i=hitsave[j];
	if(sim.random)
	    k=rorder[j];
	else
	    k=i;

	evi = evsim + i;
	evk = evhead.ev + k;

	if(reac.mass[0] == 0.0)
	    evs_m(evk, ev_m(evi));
	else
	    evs_m(evk, reac.mass[0]);
	
	evs_det(evk, ev_det(evi));	
	evs_seg(evk, 0);
	evs_fold(evk, 1);

	/*Calculate energy deposited in detector for punchthru particle*/ 
	if(sim.det_pthru){
	    evs_e(evk, reac.edisp*scanga_energy_loss(i));
	}
	else
	    evs_e(evk, reac.edisp*ev_e(evi));
	
	/*can only do this for particular detector arrangements*/
	if(sim.det_sim){
	    /*smear particle energy according to detector energy resolution
	      and set dispersion*/
	    if(sim.eres > 0)
		evs_e(evk, reac.edisp*grand(ev_e(evk)/reac.edisp,sim.eres));

	    evhead.pmult[ev_det(evk)]++;

	    /*smear particle vectors according to detector pos resolution*/
	    det_coord(ev_det(evk), &detpos, ev_pvec(evi));
	    if(sim.pres_x > 0)
		detpos.x = grand(detpos.x,sim.pres_x);
	    if(sim.pres_y > 0)
		detpos.y = grand(detpos.y,sim.pres_y);
	    evs_dir_from_coord(evk, &detpos);
	}
	else
	{
	    evs_dir_xyz(evk, ev_v_x(evi), ev_v_y(evi), ev_v_z(evi));
	}

	evs_q(evk, 0.0);
	evs_hi(evk, NULL);
	evs_li(evk, NULL);
    }
    
    return(nhits);
    
}


/*
 * Finish Scanga
 */
int scanga_finish(void)
{
    fprintf(stderr,"Scanga: Finished\n");
    return OK;
}


/*
 * Initialise detector coordinates
 */
int evsim_init(int n)
{
    if (evsim != NULL)
	free(evsim);
    
    if ((evsim = (particle *) calloc(n+1, sizeof(particle))) == NULL)
    {
	fprintf(stderr,
		"Scanga: Unable to allocate "
		"memory for simulated event structure.\n");
	return ABORT;
    }
    
    return OK;
}

/*
 * write simulated event to file or event structure
 */
int wrt_evsim(int n, FILE *fpw)
{
    int i,ntest=0;

    if(n < 1){
	fprintf(stderr,"wrt_evsim: No hit in detection system!\n");
	return OK;
    }
    
    fprintf(fpw,"%d\n",n);
    
    for(i=3; i<13; i++)
	if(sim.hit[i] > 0){
	    ntest++;
	    fprintf(fpw,"%8.3f %8.3f %8.3f\n",
		    ev_e(evsim + i), ev_p_td(evsim + i),
					  ev_p_pd(evsim + i));
	    if(ntest>n){
		fprintf(stderr,"wrt_evsim: Error! Too many hits written\n");
		fprintf(stderr,"nhits = %d ntest = %d\n",n,ntest);
		return OK;
	    }
	}
    return OK;
}

/*
 * read simulated event from file
 */
int read_evsim(void)
{
    return OK;
}


/* Random number generator with gaussian distribution
 * f(x) = 1/(s *sqrt(2*pi)) * exp(- (x-c)**2/(2*s**2))
 * with centroid c and standard deviation s
 * an inverse Chebyshev polynomial expansion is used
 * to evaulate x  = f**-1 (u) where u is the random deviate
 * see chapter 26 of Abramowitz and Stegun eqn 26.2.23
 */

#define c0 2.515517
#define c1 0.802853
#define c2 0.010328
#define d1 1.432788
#define d2 0.189269
#define d3 0.001308
#define FWHMFACT .42466090014400952136 /* (1/(2*sqrt(2*log(2))) */

double grand(double c, double fwhm)
{
    double t,tsq,n;
     
    /* assume c=0.0, s=1.0 and get random probability , 0 < t <= 0.5*/
    
    t=drand48();
    if (t >= 0.5)
    {
	n = 1;
	t -= 0.5;
    }
    else
	n = 0;

    /*compute inverse by eqn 26.2.23*/
    if (t < 1.0e-30)
    {
	t = 11.7539400024;
	tsq = 138.1550558;
    }
    else
    {
	tsq = -log(t*t);
	t=sqrt(tsq);
    }

    t -= (c0+c1*t+c2*tsq)/(1.0+d1*t+(d2+d3*t)*tsq);
    
    /*randomize x in positive and negative direction*/
    if (n)
	t = -t;
    
    /*correct for centroid and standard deviation*/
    return t*fwhm*FWHMFACT + c;
}

/*
 * random shuffle of n numbers
 */
void grorder(int m)
{
    int i,j,k;
    
    for(j=1;j<m+1;j++)  /*initialise*/
	rorder[j]=j;
    
    for(j=m; j>1; j--){
	k=j*drand48()+1;
	i=rorder[k];
	rorder[k]=rorder[j];
	rorder[j]=i;
    }
}
