/*
 * Main code for stand alone running of scanga               Steve Chappell
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "../evsubs.h"

static int tot_nevents=0;
static char *inputfile;
static char *outputfile;

int main(int argc,char **argv)
{
 int nevents,nn;
 double eff,efferr;
 FILE *ofp;
 
 /* check to see which options have been set in the command line */      
 if(read_scanga_args(argc,argv)){
     fprintf(stderr,"Scanga: Error reading command line\n"); 
     return ABORT;
 }
 
 /*Number of events for Monte Carlo*/
 if(tot_nevents < 1){
     fprintf(stderr, "Enter Nevents for simulation: ");
     if(scanf("%d",&tot_nevents) != 1){
	 fprintf(stderr,"Error reading Nevents\n");
	 return ABORT;
     }
 }
 
 /*Calculation Type for Monte Carlo*/
 fprintf(stderr, "Select Efficiency variable for Monte Carlo: \n"
	 "0: None\n"
	 "1: Ex3\n"
	 "2: Ex4\n"
	 "3: E1\n"
	 "4: Phis\n"
	 );
 if(scanf("%d",&sim.effcalc) != 1){
     fprintf(stderr,"Error reading Calc type\n");
     return ABORT;
 }

 if(sim.effcalc){
     fprintf(stderr, "Enter ranges for calculation: effmin effmax effstep: ");
     if(scanf(" %lf %lf %lf", &sim.effmin,&sim.effmax,&sim.effstep) != 3){
	 fprintf(stderr,"Error reading effmin,effmax,effstep\n");
	 return ABORT;
     }
 }
 else{
     sim.effmin=sim.effmax=sim.effstep=0.0;
 }
 
 if(scanga_init(inputfile)){
     fprintf(stderr, "Error initialising Scanga\n");
     return ABORT;
 }
 

 /*open outputfile*/
 if(outputfile == NULL)
     outputfile = "scanga_out.sca";
	 
 if ((ofp = fopen(outputfile, "w")) == NULL){	    
     fprintf(stderr, "Scanga: Error! could not open \"%s\".\n", outputfile);
     return ABORT;
 }
 fprintf(ofp,"@type xydy\n");
	 

 /*loop over Desired Efficiency Variable*/
 for(sim.effvar=sim.effmin ; sim.effvar <= sim.effmax ; 
     sim.effvar+=sim.effstep){
     
     eff = 0.0; /*reset efficiency counter*/

     if(sim.effcalc == 1){
	 sim.ex[3]=sim.effvar;
	 fprintf(stderr,"\nEx3 = %7.2f\n",sim.ex[3]);
     }
     else if(sim.effcalc == 2){
	 sim.ex[4]=sim.effvar;
	 fprintf(stderr,"\nEx4 = %7.2f\n",sim.ex[3]);
     }
     else if(sim.effcalc == 3){
	 sim.e1=sim.effvar;
	 fprintf(stderr,"\nE1 = %7.2f\n",sim.e1);
     }
     else if(sim.effcalc == 4)
	 fprintf(stderr,"\nPhis = %7.2f\n",sim.effvar);
     else if(sim.effcalc){
	 fprintf(stderr,"\nError! Can't do effcalc=%d\n",sim.effcalc);
	 return ABORT;
     }
     
     if(scanga_kin_init())
	 fprintf(stderr, "Scanga: Kinematics don't make sense ...skipping\n");
     else{
	 /*loop over sim.nevents */
	 fprintf(stderr,"\n>>: ");
	 nn=tot_nevents/50;
	 for(nevents=1; nevents <= tot_nevents; nevents++){
	     
	     /*Show progress every nn events*/
	     if (nn-- == 0)
	     {
		 nn=tot_nevents/50;
		 fprintf(stderr,"#");
	     }
	     
	     if((scanga_event() > 0)){
		 eff++; /*fractional efficiency*/
		 /*interesting event*/
	     }
	     
	 } /* End Nevents Loop */
	 
	 /*Output Efficiency in percent*/
	 if(sim.det_sim && sim.effcalc){
	     efferr=(sqrt(eff))/((double) nevents ) *100.0;
	     eff=eff/((double) nevents) *100.0;
	     fprintf(stderr,"\n\n %% Eff = %10.6f +/- %10.6f\n",
		     eff,efferr);
	     fprintf(ofp,"%f %f %f \n",sim.effvar,eff,efferr);
	 }
     }
 
 } /* End Efficiency Loop */
 
 if(scanga_finish()){
     fprintf(stderr, "Error finishing Scanga\n");
     return ABORT;
 }

 fclose(ofp);
 fprintf(stderr, "Scanga: output in file \"%s\".\n",outputfile);
 
 return OK;
}


/*
 * Read command line options for scanga
 */
int read_scanga_args(int argc, char **argv)
{
    int i; 
    
    for(i=1 ; i< argc; i++){
	if (strncmp(argv[i],"-n",2) == 0) {
	    if (argv[i+1] != NULL) {
		if(tot_nevents = (int) atoi(argv[i+1])){
		    fprintf(stderr,"Scanga: Simulation of %d events\n",
			    tot_nevents);
		    i++;
		}
		else{
		    fprintf(stderr,"Scanga: Canny simulate nevents < 1 !\n");
		    return ABORT;
		}
	    }	    
	}
	else if (strncmp(argv[i],"-i",2) == 0) {
	    if (argv[i+1] != NULL)
		inputfile = argv[i+1];
	    i++;
	}
	else if (strncmp(argv[i],"-o",2) == 0) {
	    if (argv[i+1] != NULL)
		outputfile = argv[i+1];
	    i++;
	}
	else{
	    fprintf(stderr,"Scanga: command line option "
		    "\"%s\" unrecognised\n",argv[i]);
	    fprintf(stderr, "\nUsage: scanga -n (nevents)"
		    " -i (inputfile) -o (outputfile) \n\n");
	    return ABORT;
	}
    }
    return OK;
}



