#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <time.h>

#include "sort_def.h"
#include "sort_thread.h"
#include "rdtape.h"
#include "data_io.h"
#include "spectraint.h"
#include "filt.h"

sort_IO_t 	*sort_io_data;
dirs_IO_t	*dirs_io_data;
char            *tmp_spectrum_dir;

int start_sort P((char *command));
void f_init P((void));
void initadc_ P((void));

int DBX_val;

/*ARGSUSED*/
void
read_args(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{
     int i; 
     for(i=1; argv[i] != NULL; i++) {
	   if (strncmp(argv[i],"-dbx",4) == 0) {
		 if (argv[i+1] != NULL) {
		       DBX_val = atoi(argv[i+1]);
		       if (DBX_val >= 3)
			     fprintf(stderr,"sort: diagnostics level set to %d\n",DBX_val);
		       i++;
		 }
	   }
	   else if (strncmp(argv[i],"-sdir",5) == 0) {
		 if (argv[i+1] != NULL) {
		       tmp_spectrum_dir = strdup(argv[i+1]);
		 }
		 else {
		       fprintf(stderr,"sort: no shared directory specified after -sdir option\n");
		       exit(-1);
		 }
	   }
     }
     return;
}

/* initialise the common blocks used in rdtape and the user supplied sort file in sort.f */
void
init_fort_common(
#if NeedFunctionPrototype
void
#endif
)
{
    var_str_t	**pptr;
    register int	i;
    
/* set up filenm_ common block ... file names */
    (void) memset(filenm_.filein,'\0',filenm_SIZE);
    strncpy(filenm_.filein,sort_io_data->read_file,filenm_SIZE);
    (void) memset(filenm_.fileou,'\0',filenm_SIZE);
    strncpy(filenm_.fileou,sort_io_data->write_file,filenm_SIZE);
    filenm_.runnumber = -1;

/* set up variable common block */
    pptr = get_nametab_var();
    for (i=1; i<TAB_SIZE_VAR; i++) {
	vars_.var[i-1] =  ( pptr[i] != 0) ? pptr[i]->sp->value : 0;
    }
    return;
}


/**********************************************************************************/
/*ARGSUSED*/
static void
sigterm_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 3)
	    fprintf(stderr,"sort: SIGTERM receieved...sort process exiting\n");
      exit(0);
}

/*ARGSUSED*/
static void
sigint_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 3)
	    fprintf(stderr,"sort: SIGINT receieved...\n");
      return;
}

/*ARGSUSED*/
static void
sigsegv_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 3)
	    fprintf(stderr,"sort: SIGSEGV receieved...\n");
      fprintf(stderr,"SORT ROUTINE received SIGSEGV signal ... dumping core\n");
      fprintf(stderr,"> check that you are not accessing array elements outside your array bounds\n");
      fprintf(stderr,"> or otherwise overwriting forbidden memory locations\n");
      abort();
}

/*ARGSUSED*/
static void
sigbus_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 3)
	    fprintf(stderr,"sort: SIGBUS receieved...\n");
      fprintf(stderr,"SORT ROUTINE received SIGBUS signal ... dumping core\n");
      fprintf(stderr,"> check that you are not passing an incorrect number or type\n");
      fprintf(stderr,"> of arguments in your subroutine calls\n");      
      abort();
}

/*ARGSUSED*/
static void
sigill_handler(
#if NeedFunctionPrototype
	   int sig)
#else
      sig, code, scp, addr)
      int sig;
      int code;
      struct sigcontext *scp;
      char *addr;
#endif
{
      if (DBX_val >= 3)
	    fprintf(stderr,"sort: SIGILL receieved...\n");
      fprintf(stderr,"SORT ROUTINE received SIGILL signal ... dumping core\n");
      fprintf(stderr,"> illegal instruction ??? how did you manage this?\n");
      abort();
}

/************************************************************************************************/

int
setup_sighandlers(
#if NeedFunctionPrototype
		  void)
#else
      )
#endif
{
      struct sigaction newvec;

      (void) sigemptyset(&newvec.sa_mask);

/* set up signal handler for SIGINT */
#ifdef SVR4	    
      newvec.sa_flags = SA_RESTART;
#else
      newvec.sa_flags = 0;
#endif
      (void) sigemptyset(&newvec.sa_mask);
      newvec.sa_handler = sigint_handler;
      if (sigaction(SIGINT,&newvec,NULL) == -1) {
	    fprintf(stderr,"sort: failed to setup handler for SIGINT\n");
	    return(-1);
      }
      
/* setup handler for SIGTERM */
      newvec.sa_handler = sigterm_handler;
      if (sigaction(SIGTERM,&newvec,NULL) == -1) {
	    fprintf(stderr,"sort: failed to setup handler for SIGTERM\n");
      	    return(-1);
      }

/* setup handler for SIGSEGV */
      newvec.sa_handler = sigsegv_handler;
      if (sigaction(SIGSEGV,&newvec,NULL) == -1) {
	    fprintf(stderr,"sort: failed to setup handler for SIGSEGV\n");
	    return(-1);
      }
      
/* setup handler for SIGBUS */
      newvec.sa_handler = sigbus_handler;
      if (sigaction(SIGBUS,&newvec,NULL) == -1) {
	    fprintf(stderr,"sort: failed to setup handler for SIGBUS\n");
	    return(-1);
      }

/* setup handler for SIGILL */
      newvec.sa_handler = sigill_handler;
      if (sigaction(SIGILL,&newvec,NULL) == -1) {
	    fprintf(stderr,"sort: failed to setup handler for SIGILL\n");
	    return(-1);
      }
      
      return(0);
}
      
/**************************************************************************************************/

void
main(
#if NeedFunctionPrototype
     int		argc,
     char		**argv)
#else
      argc, argv)
      int		argc;
      char		**argv;
#endif
{
      char buffer[BUFSIZ];
      int ioerr, i;

/* reset signal to default condition initially */      
      for(i=NSIG; i--; )
	    signal(i, SIG_DFL);
      
/* check to see which options have been set in the command line */      
      read_args(argc,argv);

/* set up signal handlers  */
      if (setup_sighandlers() == -1) {
	    fprintf(stderr,"sort: failed to setup signal handlers...exiting\n");
	    exit(-1);
      }
      
/* create shared data resources */
      if (tmp_spectrum_dir == NULL) {
	    fprintf(stderr,"sort: no directory for shared data file specified...exiting\n");
	    exit(-1);
      }
      else
	    sprintf(buffer,"%s/.shared.dat",tmp_spectrum_dir);
	    
      if (connect_to_shared_data(buffer, argv[0]) == -1) {
	    fprintf(stderr,"sort: error occured during connection to shared data...exiting\n");
	    exit(-1);
      }
      
/*  read in spectra definitions and create storage */
      if ( read_spectra_data() == SORT_FAIL) {
	    fprintf(stderr,"sort: error occured during reading spectra file...exiting\n");
	    exit(-1);
      }

/*  initialise fortran i/o */
      f_init();

/* command loop */      
      while( (ioerr = read(SORT_SERVER_INFD, buffer, BUFSIZ)) != -1) {
	    if (DBX_val >= 7)
		  fprintf(stderr,"sort: => %s\n",buffer);
	    
	    /* sort data in shared data file */
	    if (strncasecmp(buffer,"sort",4) == 0) {
		  if ( start_sort( strstr(buffer,"sort")) ) {
			fprintf(stderr,"sort: error detected during processing of sort command\n");
			exit(1);
		  }
	    }
	    /* update internal list of spectra */
	    else if (strncasecmp(buffer,"check spectra",13) == 0) {
		  if (remake_spectra()) {
			fprintf(stderr,"sort: error updating spectra information in sort process...exiting\n");
			exit(2);
		  }
	    }
	    /* clear internal list of spectra and remake */
	    else if (strncasecmp(buffer,"reload spectra",14) == 0) {
		  if (delete_spectra()) {
			fprintf(stderr,"sort: error modifying spectra information in sort process...exiting\n");
			exit(3);
		  }
		  if (read_spectra_data() == SORT_FAIL) {
			fprintf(stderr,"sort: error occured during reading spectra file...exiting\n");
			exit(4);
		  }
	    }	    
	    else
		  fprintf(stderr,"sort: unrecognised command %s\n",buffer);
      }
      if (ioerr == -1)
	    perror("sort read");
      exit(0);
}

/**************************************************************************************************/    
/*
 *  start_sort: function to initialise the sort and choose
 *  event decoder and encoder (only used if sortf is 1st argument
 *  specified in command string)
 *
 */

/* list of supported formats */
static char *fmt_data[FMT_ARRAY_SIZE] = {"none", FORMAT_NAME_LIST};
/* array of pointer to function returning pointer to filtdat_t */
filtdat_t *(*fmt_filter[FMT_ARRAY_SIZE])() = {FORMAT_FILTER_LIST};
/* array of pointer to function returning int */
int (*fmt_rdtape[FMT_ARRAY_SIZE])() = {FORMAT_READ_LIST};

/* translate format name into number */
static int
str_2_fmt(
#if NeedFunctionPrototype
	  char *fmtstring)
#else
      fmtstring)
      char *fmtstring;
#endif
{
      register int i;
      for(i=0; i<FMT_ARRAY_SIZE; i++) {
	    if (strcasecmp(fmtstring, fmt_data[i]) == 0)
		  return(i);
      }
      return(-1);
}

/*
 * start the sorting going by choosing encoder and decoder
 * expect command line:
 *     sort encoder decoder
 * or
 *     sortf encoder decoder
 */
int
start_sort
#if NeedFunctionPrototype
(char *command)
#else
(command)
char *command;
#endif
{
    int outfmt, infmt;
    char *token = strtok(command," ");
    int filter = (command[4] == 'f') ? 1:0;
    int int_filter = 0;
    
    /* initialize vars common block */
    init_fort_common();
    
    /* initialize data block exchange */
    init_blockio();
    
    /* initialize adcnum common block */
    (void) initadc_();
    
    /* pick up the name of the encoder */
    if ( (token = strtok(NULL, " ")) == NULL) {
	fprintf(stderr,"start_sort: no event encoder specified\n");
	return(-1);
    }
    
    /* set up encoding data stream */
    if ( (outfmt = str_2_fmt(token)) >= 0) {
	if (DBX_val >= 3) fprintf(stderr,"sort: using %s format encoder\n",
				  fmt_data[outfmt]);
	if ( init_filt(filter, fmt_filter[outfmt]()) )
	    return(-1);
    } else {
	fprintf(stderr,"start_sort called with unknown filter format\n");
	fprintf(stderr,"** no event filtering will take place **\n\n");
	filter = 0;
    }
    
    /* pick up next token - the decoders name */
    if ( (token = strtok(NULL, " ")) == NULL) {
	fprintf(stderr,"start_sort: no event decoder specified\n");
	return(-1);
    }
    infmt = str_2_fmt(token);
    
    /* check to see if we are using external or internal filtering */
    if (filter)
	int_filter = (infmt == outfmt);
    
    /* select decoding function */
    if (infmt >= 0) {
	if (DBX_val >= 3) fprintf(stderr,"sort: using %s format decoder\n",
				  fmt_data[infmt]);
	(void) fmt_rdtape[infmt](int_filter);
    } else {
	fprintf(stderr,"sort: using NO event format decoder\n");
	(void) rdtape_none(int_filter);
    }
    
    /* signal end of data */
    (void) software_eod();
    
    /* write partially filled filter record to output stream */
    if (filter) 
	(void) finish_filt();
    
    /* ensure spectra are completely up to date */
    sync_spectra();
    
    return(0);
}

/* Stick these common messages out of the way to save repetition and to
   save memory */

const char
    initfailmess[] = "sort: Error in user's init routine, aborting sort.\n",
    sortinfailmess[] =
        "sort: Error in user's sortin routine, aborting sort.\n",
    finishfailmess[] = "sort: Error in user's finish routine.\n";

