/* file datadisk.c
 *
 * prev modified 1/5/95
 * last modified 27/7/95
 *
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include "sort_def.h"
#include "data_disk.h"

#define MAXBUF 16384
#define READ_MODE  0
#define WRITE_MODE 1
#ifndef SORT_TRUE
#define SORT_TRUE 1
#endif

struct header {
	union {
		short length;
		unsigned char  b[2];
	}h;
	union {
		short filler;
		unsigned char  b1[2];
	}f;
};

static int ddread_vr P((int, char *));
static int ddread_fr P((int, char *));
static int ddwrite_vr P((int, char *));
static int ddwrite_fr P((int, char *));
static int ddflush P((void));

struct disk_info {
      int          fd;
      char         *file_name;
      int          file_open;
      int          blk_len;
      int          rec_len;
      IOfunc       iofunc;
};


/* read data */
static struct disk_info read_dev;
static struct header r_bufhead,r_rechead;
static unsigned char r_iobuf[MAXBUF];
static unsigned char *r_curptr;
static int r_counter = 0;

/* write data */
static struct disk_info write_dev;
static struct header w_bufhead,w_rechead;
static char w_iobuf[MAXBUF];
static char *w_curptr;
static int  w_counter = 0;

/*
 *  open disk file for data reading
 *
 *  use specified block length or preferred unix disk block size from stat(2) call
 *  relcen = 0 indictaes that we using variable length records (gec)
 *  and reclen != 0 for fixed length records (no block/record headers)
 */
int
ddopen(
#if NeedFunctionPrototype
	   char   *file_name,
	   int    access_mode,
           int    blocklen,
           int    reclen)
#else
      file_name, access_mode, blocklen, reclen)
      char    *file_name;
      int     access_mode;
      int     blocklen;
      int     reclen;
#endif
{
      struct stat   buf;

      if (access_mode == READ_MODE) {
	    if (read_dev.file_open == SORT_TRUE) {
		  fprintf(stderr,"DDOPEN: read file already open \n");
		  return(-1);
	    }
	    /* check that files exists */
	    if (stat(file_name,&buf)) {
		  perror("DDOPEN");
		  return(-1);
	    }
/*
	    if ( (! S_ISREG(buf.st_mode)) && (! S_ISLNK(buf.st_mode)) ) {
		  fprintf(stderr,"%s is not a regular file!\n",file_name);
		  return(-1);
	    }
*/
	    /* specify block size used in read */
	    if (! blocklen){
		  read_dev.blk_len = (buf.st_blksize <= MAXBUF) ? buf.st_blksize : MAXBUF;
	    }
	    else if (blocklen > MAXBUF) {
		  fprintf(stderr,"DDOPEN: block %d size too large\n",blocklen);
		  return(-1);
	    }
	    else {  
		  read_dev.blk_len = blocklen;
	    }
	    /* specify record length of data */
	    if (! reclen){
		  read_dev.rec_len = 0;
		  read_dev.iofunc = ddread_vr;
	    }
	    else {
		  if (reclen > MAXBUF || reclen > read_dev.blk_len) {
			fprintf(stderr,"DDOPEN: record len %d too large\n",reclen);
			return(-1);
		  }
		  read_dev.rec_len = reclen;
		  read_dev.iofunc = ddread_fr;
	    }
	    /* open disk file for reading */
	    if ((read_dev.fd = open(file_name,O_RDONLY)) == -1) {
		  perror("DDOPEN");
		  return(-1);
	    }
	    read_dev.file_open = SORT_TRUE;
	    read_dev.file_name = strdup(file_name);
	    r_counter = 0;
      }
      else if (access_mode == WRITE_MODE) {
	    if (write_dev.file_open == SORT_TRUE) {
		  fprintf(stderr,"DDOPEN: write file already open \n");
		  return(-1);
	    }
	    /* check that file doesn't already exist */
	    if (access(file_name, W_OK) == -1) {
		  if (errno != ENOENT) {
			perror("DDOPEN");
			return(-1);
		  }
	    }
            else {
		  fprintf(stderr,"DDOPEN file %s already exists\n",file_name);
		  return(-1);
	    }
	    if ((write_dev.fd = open(file_name,O_WRONLY|O_CREAT,0755)) == -1) {
		  perror("DDOPEN");
		  return(-1);
	    }
	    if (stat(file_name,&buf)) {
		  (void) close(write_dev.fd);
		  perror("DDOPEN");
		  return(-1);
	    }
	    /* specify block length of data */
	    if (! blocklen || blocklen > MAXBUF)
		  write_dev.blk_len = (buf.st_blksize < MAXBUF) ? buf.st_blksize : MAXBUF;
	    else
		  write_dev.blk_len = blocklen;
	    /* specify record length of data */
	    if (! reclen){
		  write_dev.rec_len = 0;
		  /* set up header field for first block */
		  w_curptr = w_iobuf + 4;
		  w_bufhead.h.length = 4;
		  w_bufhead.f.filler = (short) 0x8020;
		  write_dev.iofunc = ddwrite_vr;
	    }
	    else {
		  if (reclen > MAXBUF || reclen > write_dev.blk_len) {
			fprintf(stderr,"DDOPEN: record len %d too large\n",reclen);
			(void) close(write_dev.fd);
			return(-1);
		  }
		  write_dev.rec_len = reclen;
		  write_dev.iofunc = ddwrite_fr;
		  w_curptr = w_iobuf;
	    }
	    write_dev.file_open = SORT_TRUE;
	    write_dev.file_name = strdup(file_name);
	    w_counter = 0;
      }
      else {
	    fprintf(stderr,"DDOPEN: unrecognised access mode\n");
	    return(-1);
      }
      return(0);
}

/*
 *  return pointer to function that will perform I/O operation on disk file
 */
IOfunc
ddiofunc(
#if NeedFunctionPrototype
	 int access_mode)
#else
	  access_mode)
          int access_mode;
#endif
{
      if (access_mode == READ_MODE) {
	    return(read_dev.iofunc);
      }
      else if (access_mode == WRITE_MODE) {
	    return(write_dev.iofunc);
      }
      else 
	    fprintf(stderr,"DDIOFUNC: unrecognised access mode\n");
      return(NULL);
}


/*
 *   
 *   print out some of the information held in the file_data structure
 *   for either the read or write OPEN tape file
 */
int
ddstate(
#if NeedFunctionPrototype
	int access_mode)
#else
      access_mode)
      int access_mode;
#endif
{
      if (access_mode == READ_MODE) {
	    if( ! read_dev.file_open )
		  return(1);
	    fprintf(stderr,"Reading from disk file %s \nwith blocklen = %d and ",
		    read_dev.file_name, read_dev.blk_len);
	    if (read_dev.rec_len == 0)
		  fprintf(stderr, "variable reclen\n");
	    else
		  fprintf(stderr, "fixed reclen = %d\n",read_dev.rec_len);
	    return(0);

      }
      else if (access_mode == WRITE_MODE) {
	    if( ! write_dev.file_open )
		  return(1);
	    fprintf(stderr,"Writing to disk file %s \nwith blocklen = %d and ",
		    write_dev.file_name, write_dev.blk_len);
	    if (write_dev.rec_len == 0)
		  fprintf(stderr, "variable reclen\n");
	    else
		  fprintf(stderr, "fixed reclen = %d\n",write_dev.rec_len);
	    return(0);
      }
      else
	    fprintf(stderr,"DTSTATE: unrecognised access mode\n");
      return(-1);
}

/*
 * close disk file for data reading
 */
int
ddclose(
#if NeedFunctionPrototype
	int access_mode)
#else
      access_mode)
      int  access_mode;
#endif
{
      if (access_mode == READ_MODE) {
	    if (read_dev.file_open != SORT_TRUE)  {
		  fprintf(stderr,"DDCLOSE: read file not open \n");
		  return(-1);
	    }
	    if ( close(read_dev.fd) == -1) {
		  perror("DDCLOSE");
		  return(-1);
	    }
	    read_dev.file_open = 0;
	    if (read_dev.file_name != NULL) {
		  (void) free(read_dev.file_name);
	    }
      }
      else if (access_mode == WRITE_MODE) {
	    if (write_dev.file_open != SORT_TRUE) {
		  fprintf(stderr,"DDCLOSE: write file not open \n");
		  return(-1);
	    }
	    if ( ddflush() != 0) {
		  fprintf(stderr,"DDCLOSE: error flushing filter file\n");
		  return(-1);
	    }
	    if ( close(write_dev.fd) == -1) {
		  perror("DDCLOSE");
		  return(-1);
	    }
	    write_dev.file_open = 0;
	    if (write_dev.file_name != NULL) {
		  free(write_dev.file_name);
	    }
      }
      else {
	    fprintf(stderr,"DDCLOSE: unrecognised access mode\n");
	    return(-1);
      }
      return(0);
}


/*
   int ddread_vr (maxbuflen,buffer)
   int  maxbuflen;         maximum size of buffer
   char *buffer[];         pointer to buffer containing data

   variable length records (gec format)
 */
static int
ddread_vr(
#if NeedFunctionPrototype
	    int           maxbuflen,
	    char          *buffer)
#else
      maxbuflen,buffer)
      int   maxbuflen;
      char  *buffer;
#endif
{
	int reclen;
	
/* if no more data left in local buffer read another block */      
	if (r_counter <= 0) {
		r_curptr = r_iobuf;
		if ((r_counter = read(read_dev.fd, r_iobuf, read_dev.blk_len)) <= 0){
		    if (r_counter < 0)
			  perror("ddread_vr error");
		    return (r_counter);
		}
		r_bufhead.h.b[0] = *r_curptr++;
		r_bufhead.h.b[1] = *r_curptr++;
		r_curptr += 2;
		r_counter -= 4;
		r_bufhead.h.length -= 4;
	}
	r_rechead.h.b[0] = *r_curptr++;
	r_rechead.h.b[1] = *r_curptr++;
	r_curptr += 2;
	r_counter -= 4;
	reclen = r_rechead.h.length - 4;

/* check buffer size large enough for complete record */
	if (maxbuflen < reclen) {
	      fprintf(stderr,"DDREAD: buffer size %d less than record size of data %d\n",maxbuflen,reclen);
	      return(-1);
	}

/* copy data from local buffer into users buffer */	
	(void) memcpy(buffer, r_curptr, reclen);

	r_curptr += reclen;
	r_counter -= reclen;
	if ( (r_bufhead.h.length -= r_rechead.h.length) <= 0)
		r_counter = 0;
	return (reclen);
}


/*
   int ddread_fr (maxbuflen,buffer)
   int  maxbuflen;         maximum buffer length 
   char *buffer[];         pointer to buffer containing data

   fixed length records
   */
static int
ddread_fr(
#if NeedFunctionPrototype
	    int           maxbuflen,
	    char          *buffer)
#else
      maxbuflen,buffer)
      int   maxbuflen;
      char  *buffer;
#endif
{
/* if no more data left in local buffer read another block */      
	if (r_counter <= 0) {
		r_curptr = r_iobuf;
		if ((r_counter = read(read_dev.fd, r_iobuf, read_dev.blk_len)) <= 0){
		    if (r_counter < 0)
			  perror("ddread_fr error");
		    return (r_counter);
		}
	}
	
/* check buffer size large enough for complete record */
	if (maxbuflen < read_dev.rec_len) {
	      fprintf(stderr,"DDREAD: buffer size %d less than record size of data %d\n",
		      maxbuflen,read_dev.rec_len);
	      return(-1);
	}

/* copy data from local buffer into users buffer */
	(void) memcpy(buffer, r_curptr, read_dev.rec_len);

	r_curptr += read_dev.rec_len;
	r_counter -= read_dev.rec_len;
	return (read_dev.rec_len);
}


/*
  int ddwrite_vr(len,buffer)
  int len			length to put
  char buffer[]	                buffer to put
  VARIABLE LENGTH RECORDS
 */
static int
ddwrite_vr(
#if NeedFunctionPrototype
	    int  len,
	    char buffer[])
#else
      len,buffer)
      int  len;
      char buffer[];
#endif
{
	int ierr;

	if (len > write_dev.blk_len-8)
	      return(-1);
	
/* if not enough room left in buffer, write buffer to disk */
	if ((write_dev.blk_len -w_bufhead.h.length -4) < len){
		w_iobuf[0] = w_bufhead.h.b[0];
		w_iobuf[1] = w_bufhead.h.b[1];
		w_iobuf[2] = w_bufhead.f.b1[0];
		w_iobuf[3] = w_bufhead.f.b1[1];
		if ((ierr = write(write_dev.fd,w_iobuf,write_dev.blk_len)) <= 0){
			if (ierr < 0) perror("ddwrite_vr error");
			return (ierr);
		}
		w_curptr = w_iobuf + 4;
		w_bufhead.h.length = 4;
		w_bufhead.f.filler = (short) 0x8020;
	}
	
/* fill in record header information */
	w_rechead.h.length = len+4;
	w_rechead.f.filler = 0;
	*w_curptr++ = w_rechead.h.b[0];
	*w_curptr++ = w_rechead.h.b[1];
	*w_curptr++ = w_rechead.f.b1[0];
	*w_curptr++ = w_rechead.f.b1[1];
	
/* copy data to local buffer */
	(void) memcpy( w_curptr, buffer, len);

	w_bufhead.h.length += w_rechead.h.length;
	w_curptr += len;

	return (len);
}


/*
  int ddwrite_fr(len,buffer)
  int len			length to put
  char buffer[]	                buffer to put
  FIXED LENGTH RECORDS
 */
static int
ddwrite_fr(
#if NeedFunctionPrototype
	    int  len,
	    char buffer[])
#else
      len,buffer)
      int  len;
      char buffer[];
#endif
{
	int ierr;
	
	if (len > write_dev.blk_len)
	      return(-1);

/* len == block length write data straight to disk ... no blocking */
	if ( len == write_dev.blk_len) {
	      if ((ierr = write(write_dev.fd, buffer, write_dev.blk_len)) != len){
		    if (ierr < 0)
			  perror("write_fr error");
		    else if (ierr > 0) {
			  fprintf(stderr,"write_fr error: wrote incomplete block to disk\n");
			  ierr = -ierr;
		    }
		    return (ierr);
	      }
	}
	else {
	      /* check to see if enough space left in local buffer .. if not write buffer out */
	      if ((write_dev.blk_len - w_counter) < len) {
		    if (w_counter < write_dev.blk_len)
			  (void) memset(w_curptr, 0, (write_dev.blk_len - w_counter));
		    if ((ierr = write(write_dev.fd, w_iobuf, write_dev.blk_len)) <= 0){
			  if (ierr < 0)
				perror("write_fr error");
			  return (ierr);
		    }
		    w_curptr = w_iobuf;
		    w_counter = 0;
	      }
	      /* copy data to local buffer */
	      (void) memcpy( w_curptr, buffer, len);
	      w_counter += len;
	      w_curptr += len;
	}
	
	return (len);
}

/*
  int ddflush(void)
  flushes incomplete write data buffer
 */ 
static int
ddflush(
#if NeedFunctionPrototype
	    void)
#else
      )
#endif
{
	int ierr;

/* check to see if data to written in record structure */	
	if (write_dev.rec_len == 0) {
	      if (w_bufhead.h.length > 4) {
		    w_iobuf[0] = w_bufhead.h.b[0];
		    w_iobuf[1] = w_bufhead.h.b[1];
		    w_iobuf[2] = w_bufhead.f.b1[0];
		    w_iobuf[3] = w_bufhead.f.b1[1];
		    if ((ierr = write(write_dev.fd,w_iobuf,write_dev.blk_len)) <= 0){
			  if (ierr < 0) perror("ddflush error");
			  return (ierr - 1);
		    }
		    w_curptr = w_iobuf + 4;
		    w_bufhead.h.length = 4;
		    w_bufhead.f.filler = (short) 0x8020;
	      }
	}
	else {
	      if (w_counter < write_dev.blk_len)
		    (void) memset(w_curptr, 0, (write_dev.blk_len - w_counter));
	      if (w_counter > 0) {
		    if ((ierr = write(write_dev.fd, w_iobuf, write_dev.blk_len)) <= 0){
			  if (ierr < 0) perror("ddflush error");
			  return (ierr);
		    }
	      }
	}
	return (0);
}
	


