/*
 *  loading and saving interface routines
 *  call libary functions to perform SDB & EG operations
 *  C ordering adopted for 2d spectrum arrays
 *
 *  last modified 22/11/95
 *
 */
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>

#include "sort_def.h"
#include "sort_threadint.h"
#include "read_spec.h"
#include "discsubs.h"
#include "sort_main.h"
#include "sdb.h"
#include "eg.h"

void swap2d_c2f P((int *, int *, int, int));
int Save_selected_1d P((dirs_IO_t *, char *));
int Save_selected_2d P((dirs_IO_t *, char *));

int
Save_1d(
#if NeedFunctionPrototype
	spec_1d_t	*ptr_1d,
	char		*save_dir,
	int		mode,		
        int             type)   
#else
      ptr_1d,save_dir,mode,type)
      spec_1d_t	        *ptr_1d;	/* pointer to spectrum data structure */
      char		*save_dir;	/* directory name to which data is to be saved */
      int		mode;		/* true if over writting of files is permitted */
      int               type;           /* specify the format of the save file */
#endif
{
	char		sbuf[MED_SUNSORT_BUFFER+32];	        /* file name buffer */
	char		*name = ptr_1d->sp->name;		/* spectrum name */
	int		nos = ptr_1d->nos;		        /* spectrum storage number */
	int		size = ptr_1d->sp->size;		/* spectrum size */
	int		*data = ptr_1d->data;		        /* pointer to spectrum data */
	int		id;				        /* sdb file id number */

	FILE            *fp; /* added to write X vs Y format file */
	register int    x;
	int             error = 0;
	
/* generate file name of spectrum to be saved */
	sprintf(sbuf,"%s/%s_1d_%d",save_dir,name,nos);
	
/* inquire whether file already exists and check whether ok to over write */
	if ( inquire(sbuf)) {
		if (mode) {
			if ( unlink(sbuf)) {
				perror("Save_1d");
				return(SORT_FAIL);
			}
		}
		else {
			fprintf(stderr,
				"server: file %s already exists and not in overwrite mode\n",sbuf);
			return(0);
		}
	}

	switch(type) {
/* write data out to new sdb file */
	   case SDB_TYPE:
	      if ( ! sdef(&id,0,size,0,0,4,sbuf,strlen(sbuf)) ) {
		    if ( swrit(id,0,size,0,0,data,4))
			  error = 1;
		    if ( sname(&id,sbuf,0) || error) 
			  return(SORT_FAIL);
	      }
	      else
		    return(SORT_FAIL); 
	      break;

/* write data out to text file in X vs Y format */
           case XY_TYPE:
	      if ( (fp = fopen(sbuf,"w")) == NULL) {
		    fprintf(stderr,"server: failed to open %s during save\n",sbuf);
		    perror("sort");
		    return(SORT_FAIL);
	      }
	      fprintf(fp,"#X-Y 1D spectrum named %s\n",name);
	      for (x=0; x < size; x++) 
		    fprintf(fp,"%d %d\n",x,data[x]);

	      (void) fclose(fp);
	      break;

	   case EG_TYPE:
              eexpt(&id);
	      if ( id != 0) {
	            fprintf(stderr,"server: failed to connect EG experiment\n");
		    return(SORT_FAIL);
	      }
              if ( ! edef(&id,0,size,0,0,4,sbuf,strlen(sbuf)) ) {
		    if ( ewrit(id,0,size,0,0,data,4)) 
			  error = 1;
		    if ( ename(&id,sbuf,0) || error) 
			  return(SORT_FAIL);
	      }
	      else
		    return(SORT_FAIL); 
	      break;
	      
	   default:
	      fprintf(stderr,"server: file type 0x%x specified for save unrecognised\n",type);
	      return(SORT_FAIL);
	}

	fprintf(stderr,"server: saved 1d spectra number %d as %s\n",nos,sbuf);

	return(0);
}
int
Save_2d(
#if NeedFunctionPrototype
	spec_2d_t	*ptr_2d,
	char		*save_dir,
	int		mode,
        int             type)
#else
      ptr_2d,save_dir,mode,type)
      spec_2d_t	        *ptr_2d;		/* see comments for 1d case */
      char		*save_dir;
      int		mode;
      int               type;
#endif
{
	char		sbuf[MED_SUNSORT_BUFFER+32];
	char		*name = ptr_2d->sp->name;
	int		nos = ptr_2d->nos;
	int		size = ptr_2d->sp->size;
	int		*data = ptr_2d->data;
	int		id;


	FILE            *fp; /* added to write X vs Y format file */
	register int    x,y;
	int             *ptr;
	int             error = 0;


/* generate file name of spectrum to be saved */
	sprintf(sbuf,"%s/%s_2d_%d",save_dir,name,nos);
	
/* inquire whether file already exists and check whether ok to over write */
	if ( inquire(sbuf)) {
		if (mode) {
			if ( unlink(sbuf)) {
				perror("Save_2d");
				return(SORT_FAIL);
			}
		}
		else {
			fprintf(stderr,

				"server: file %s already exists and not in overwrite mode\n",sbuf);
			return(0);
		}
	}

	switch(type) {

	   case SDB_TYPE:
/* swap data to FORTRAN ordering */	      
	      ptr = (int *) calloc(size*size, sizeof(int));
	      if (ptr == NULL) 
		    return(SORT_FAIL);
	      swap2d_c2f(ptr,data,size,size);
/* write data out to new sdb file */
	      if ( ! sdef(&id,0,size,0,size,4,sbuf,strlen(sbuf)) ) {
		    if ( swrit(id,0,size,0,size,ptr,4) )
			  error = 1;
		    (void) free(ptr);
		    if ( sname(&id,sbuf,0) || error)
			  return(SORT_FAIL);
	      }
	      else {
		    (void) free(ptr);
		    return(SORT_FAIL);
	      }
	      break;

/* write out to text file in X vs Y format */
	   case XY_TYPE:
	      if ( (fp = fopen(sbuf,"w")) == NULL) {
		    fprintf(stderr,"server: failed to open %s during save\n",sbuf);
		    perror("sort");
		    return(SORT_FAIL);
	      }
	      fprintf(fp,"#X-Y 2D spectrum named %s\n",name);
	      for (y=0; y < size; y++)
		    for (x=0; x < size; x++) 
			  fprintf(fp,"%d %d %d \n",x,y,data[y + x*size]);  /* C ordering */

	      (void) fclose(fp);
	      break;

	   case EG_TYPE:
/* write data out to new eg file */
	      eexpt(&id);
	      if ( id != 0) {
	            fprintf(stderr,"server: failed to connect EG experiment\n");
		    return(SORT_FAIL);
	      }
	      if ( ! edef(&id,0,size,0,size,4,sbuf,strlen(sbuf)) ) { 
		    if ( ewrit(id,0,size,0,size,data,4) )
			  error = 1;
		    if ( ename(&id,sbuf,0) || error) 
			  return(SORT_FAIL);
	      }
	      else {
		    return(SORT_FAIL);
	      }
	      break;
	      
	   default:
	      fprintf(stderr,"server: file type specified for save unrecognised\n");
	      return(SORT_FAIL);
	}

	fprintf(stderr,"server: saved 2d spectra number %d as %s\n",nos,sbuf);
	
	return(0);
}

/*
 * Inquire about existance of file. Return SORT_FALSE=0 if file doesnot exist
 * or failure, else return non zero value depending on file type.
 */
int
inquire(
#if NeedFunctionPrototype
	char 		*file_name)
#else
      file_name)
      char 		*file_name;
#endif
{
	struct stat	buf;
	
	if ( stat(file_name,&buf) == 0) {
		if (S_ISREG(buf.st_mode)) return(SORT_reg);
		if (S_ISDIR(buf.st_mode)) return(SORT_dir);
		if (S_ISLNK(buf.st_mode)) return(SORT_lnk);
		return(SORT_TRUE);
	}
	else {
		if (errno != ENOENT) perror("sort inquire");
		return(SORT_FALSE);
	}
}

int
Save_all(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      spec_1d_t		*p1d;
      spec_2d_t		*p2d;
      int               overwrite = (ptr->IO_state & OVER_WRITE);
      int               format = (ptr->IO_state & FMT_MASK);

/* get pointers to spectrum storage tables */
      p1d = get_nametab_1d()[0];
      p2d = get_nametab_2d()[0];

/* check whether dirctory exists and set over write mode */
      if (Query_Save_directory(ptr))
	    return(SORT_FAIL);

/* save all the one dimensional spectra */
      while ( p1d ) {
	    if (Save_1d(p1d, ptr->save_dir, overwrite, format)) {
		  return(SORT_FAIL); 
	    }
	    p1d = p1d->next;
      }
/* save all the two dimensional spectra */
      while ( p2d ) {
	    if (Save_2d(p2d, ptr->save_dir, overwrite, format)) {
		  return(SORT_FAIL);
	    }
	    p2d = p2d->next;
      }
      return(SORT_TRUE);
}

int
Save_windows(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      spec_2d_t		*p2d;
      int               overwrite = (ptr->IO_state & OVER_WRITE);
      int               format = (ptr->IO_state & FMT_MASK);

/* get pointers to spectrum storage tables */
      p2d = get_nametab_2d()[0];

/* check whether dirctory exists and set over write mode */
      if (Query_Save_directory(ptr))
	    return(SORT_FAIL);

/* save all the two dimensional spectra */
      while ( p2d ) {
	    if (p2d->win)
		  if (Save_2d(p2d, ptr->save_dir, overwrite, format)) {
			return(SORT_FAIL);
		  }
	    p2d = p2d->next;
      }
     return(SORT_TRUE);
}

int
Save_variables(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      var_str_t		        **ppvar, *pvar;
      FILE			*fp;
      char                      sbuf[BUFSIZ];
      
/* get pointer to variable storage table */
      ppvar = get_nametab_var();
      pvar = ppvar[0];
     
/* check whether dirctory exits and set over write mode */
      if (Query_Save_directory(ptr))
	    return(SORT_FAIL);
      
      sprintf(sbuf,"%s/.variables",ptr->save_dir);
/* check to see if file already exists and if it does whether we should preserve it */
      if ( inquire(sbuf) && !(ptr->IO_state & OVER_WRITE) ) {
	    char sbuf2[MED_SUNSORT_BUFFER+32];
	    sprintf(sbuf2,"%s/..variables",ptr->save_dir);
	    (void) rename(sbuf,sbuf2);
      }
/* open output file "variables" */
      if ( (fp = fopen(sbuf,"w")) == NULL) {
	    perror("server Save Variables");
	    return(SORT_FAIL);
      }
/* save all the variables */
      while ( pvar ) {
	    fprintf(fp,"%d   %s   %g\n",pvar->nos,pvar->sp->name,pvar->sp->value);
	    pvar = pvar->next;
      }
      fclose(fp);
      return(SORT_TRUE);
}

void
adjust_white_space(string)
      char  *string;
{
      register i;
      int      len = strlen(string);
      for(i=len; i>=0; i--) {
	    if ( isspace((int) string[i]))
		string[i] = ' ';
      }
      return;
}

int
Save_selected(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      int               fd;
      int               ret_code;
      char              buf[BUFSIZ];
      char              *token;
      char              *str_pos;

/* check whether dirctory exists and set over write mode */
      if (Query_Save_directory(ptr))
	    return(SORT_FAIL);

/* request a list of which spectra are to be saved */
      write_reply(SORT_TRUE,"Enter list of spectrum numbers in format:\n1d 1 2 3 6\n2d 1 4 7\n",-1);

/* loop reading input until save completed or error */
      for (ret_code = SORT_TRUE; ret_code != SORT_FAIL; ) {
	    if ( (fd = check_input()) != -1) {
		  if ( (ret_code = read_input(fd, buf)) != SORT_FAIL) {
			if (DBX_val >= 5)
			      fprintf(stderr,"server (save): %s\n",buf);
			adjust_white_space(buf);
			if ( (token = strtok_r(buf," ",&str_pos)) == NULL) 
			      break;
			else if (strncasecmp(token,"1d",2) == 0) {
			      if (Save_selected_1d(ptr,str_pos) == SORT_TRUE)
				    write_reply(SORT_TRUE,"\nsave => ",-1);
			      else
				    write_reply(SORT_TRUE,"Error on save 1d\n",-1);
			}
			else if (strncasecmp(token,"2d",2) == 0) {
			      if (Save_selected_2d(ptr,str_pos) == SORT_TRUE)
				    write_reply(SORT_TRUE,"\nsave => ",-1);
			      else
				    write_reply(SORT_TRUE,"Error on save 2d\n",-1);
			}
			else
			      fprintf(stderr,"server unrecognised command during save => %s\n",buf);
		  }
	    }
      }
      return(SORT_TRUE);
}

int
Save_selected_1d(
#if NeedFunctionPrototype
		     dirs_IO_t *ptr, char *buf)
#else
      ptr, buf)
      dirs_IO_t  *ptr;
      char                 *buf;
#endif
{
      char              *token;
      spec_1d_t		**pp1d, *p1d;
      int               spec_nos;
      int               overwrite = (ptr->IO_state & OVER_WRITE);
      int               format = (ptr->IO_state & FMT_MASK);

/* get pointer to spectrum storage tables */
      pp1d = get_nametab_1d();

      while ( (token = strtok_r(NULL, " ",&buf)) != NULL) {
	    spec_nos = atoi(token);
	    if (spec_nos > 0 && spec_nos < TAB_SIZE_1D) {
		  p1d = pp1d[spec_nos];
		  if (p1d != NULL) {
			if (Save_1d(p1d, ptr->save_dir, overwrite, format)) {
			      return(SORT_FAIL); 
			}
		  }
	    }
      }
      return(SORT_TRUE);
}
int
Save_selected_2d(
#if NeedFunctionPrototype
		     dirs_IO_t *ptr, char *buf)
#else
      ptr, buf)
      dirs_IO_t  *ptr;
      char                 *buf;
#endif
{
      char              *token;
      spec_2d_t		**pp2d, *p2d;
      int               spec_nos;
      int               overwrite = (ptr->IO_state & OVER_WRITE);
      int               format = (ptr->IO_state & FMT_MASK);

/* get pointer to spectrum storage tables */
      pp2d = get_nametab_2d();

      while ( (token = strtok_r(NULL, " ",&buf)) != NULL) {
	    spec_nos = atoi(token);
	    if (spec_nos > 0 && spec_nos < TAB_SIZE_2D) {
		  p2d = pp2d[spec_nos];
		  if (p2d != NULL) {
			if (Save_2d(p2d, ptr->save_dir, overwrite, format)) {
			      return(SORT_FAIL); 
			}
		  }
	    }
      }
      return(SORT_TRUE);
}


int
Load_all(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      DIR		*dirp;
      struct dirent	*dp;
      char              sbuf[BUFSIZ];

/* open directory for reading and loop through all enteries */
      if ((dirp = opendir(ptr->load_dir)) == NULL) {
	    perror("server Load");
	    return(SORT_FALSE);
      }
      for (dp=readdir(dirp); dp != NULL; dp=readdir(dirp)) {
	    if ( (dp->d_name)[0] == '.') 
		  continue; /* skip . .. and .variables*/
	    (void) sprintf(sbuf,"%s/%s",ptr->load_dir,dp->d_name);
	    if ( inquire(sbuf) == SORT_dir)
		  continue; /* skip non regular files */ 
	    if (load_spectrum_file(ptr->load_dir,dp->d_name, ptr->IO_state ) == SORT_FAIL) {
		  closedir(dirp);
		  return(SORT_FALSE);
	    }
      }
      closedir(dirp);
      return(SORT_TRUE);
}

int
Load_windows(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      DIR		*dirp;
      struct dirent	*dp;
      char              sbuf[BUFSIZ];

/* open directory for reading and loop through all enteries */
      if ((dirp = opendir(ptr->load_dir)) == NULL) {
	    perror("server Load");
	    return(SORT_FALSE);
      }
      for (dp=readdir(dirp); dp != NULL; dp=readdir(dirp)) {
	    if ( (dp->d_name)[0] == '.' || (dp->d_name)[0] != 'w') 
		  continue; /* skip . .. and .variables*/
	    (void) sprintf(sbuf,"%s/%s",ptr->load_dir,dp->d_name);
	    if ( inquire(sbuf) == SORT_dir)
		  continue; /* skip non regular files */ 
	    if (load_spectrum_file(ptr->load_dir,dp->d_name, ptr->IO_state ) == SORT_FAIL) {
		  closedir(dirp);
		  return(SORT_FALSE);
	    }
      }
      closedir(dirp);
      return(SORT_TRUE);
}

int
Load_variables(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      char			sbuf[BUFSIZ];
      var_str_t		        **ppvar, *pvar;
      FILE			*fp;
      int 			nos;
      float 			value;
      char 			name[12];

/* get pointer to variable storage table */
      ppvar = get_nametab_var();
      pvar = ppvar[0];

      sprintf(sbuf,"%s/.variables",ptr->load_dir);
/* open .variables file for reading */
      if ( (fp = fopen(sbuf,"r")) == NULL) {
	    perror("server LOAD");
	    return(SORT_FALSE);
      }
/* read file .variables and update data structures accordingly */
      while (fgets(sbuf,MED_SUNSORT_BUFFER,fp)) {
	    if (sscanf(sbuf,"%d %s %g",&nos,name,&value) != 3) {
		  if (DBX_val >= 5) {
			fprintf(stderr,"server: format error during read of .variables\n");
			fprintf(stderr,"<<%s>>\n",sbuf);
		  }
		  continue;
	    }
	    if ((pvar = ppvar[nos]) == 0) { 	/* new variable */
		  (void) make_var(nos,value,name);
	    }
	    else {				/* existing variable */
		  pvar->sp->value = value;
		  (void) strcpy(pvar->sp->name,name);
	    }
	    fprintf(stderr,"value %g load into variable #%d\n",value,nos);
      }
      fclose(fp);
      return(SORT_TRUE);
}

int
Load_selected(
#if NeedFunctionPrototype
		     dirs_IO_t	*ptr)
#else
      ptr)
      dirs_IO_t	*ptr;
#endif
{
      int               fd;
      int               ret_code;
      char              buf[BUFSIZ];
      char              *token;
      char              *str_pos;
      int               state = ptr->IO_state;
      char              *dir = ptr->load_dir;

/* request a list of which spectra are to be saved */
      write_reply(SORT_TRUE,"Enter list of file names you want loaded\n",-1);

/* loop reading input until save completed or error */
      for (ret_code = SORT_TRUE; ret_code != SORT_FAIL; ) {
	    if ( (fd = check_input()) != -1) {
		  if ( (ret_code = read_input(fd, buf)) != SORT_FAIL) {
			if (DBX_val >= 5)
			      fprintf(stderr,"server (load): %s\n",buf);
			adjust_white_space(buf);
			if ( (token = strtok_r(buf," ",&str_pos)) == NULL) 
			      break;
			else {
			      do {
				    if (load_spectrum_file(dir, token, state) == SORT_FAIL)
					  write_reply(SORT_TRUE,"Error on load\n",-1);
				    else
					  write_reply(SORT_TRUE,"\nload => ",-1);
			      } while ( (token = strtok_r(NULL, " ",&str_pos)) != NULL);
			}
		  }
	    }
      }
      return(SORT_TRUE);
}


/*
 *   swap integer data held in linear array stored in C ordering
 *   into an array with FORTRAN ordering
 */
void
swap2d_c2f(
#if NeedFunctionPrototype
	   int *fort, int *c, int sizex, int sizey)
#else
      fort, c, sizex, sizey)
      int *fort;
      int *c;
      int sizex;
      int sizey;
#endif
{
      register int x,y;

/* perform swaping */
      for (x=0; x<sizex; x++) {
	    for (y=0; y<sizey; y++) {
		  fort[x + y*sizex] = c[y + x*sizey];
	    }
      }
      return;
}

#if 0 /* This function is never used */
/*
 *   swap integer data held in linear array stored in FORTRAN ordering
 *   into an array with C ordering
 */
void
swap2d_f2c(
#if NeedFunctionPrototype
	   int *fort, int *c, int sizex, int sizey)
#else
      fort, c, sizex, sizey)
      int *fort;
      int *c;
      int sizex;
      int sizey;
#endif
{
      register int x,y;

/* perform swaping */
      for (x=0; x<sizex; x++) {
	    for (y=0; y<sizey; y++) {
		  c[y + x*sizey] = fort[x + y*sizex];
	    }
      }
      return;
}
#endif
