#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/file.h>

#include "sort_def.h"
#include "sdb.h"

struct sdb_file sd_files[NUM_FILES] = { 0 };
int flush_buf P((struct sdb_file *));
int read_rec P((int, int, int, int, void *, int, int));
int write_rec P((int, int, int, int, void *, int, int));
int update_tmp P((struct sdb_file *, int *));

int gl_currentspec = 0;

/********************************************************
 * Functions that manage the reading and writing of     *
 * records to a file. If any part of a record is        *
 * required all the record is read into memory. On      *
 * writing any part of a record only the bytes affected *
 * are changed - and eventually written out. This allows*
 * writes to be done in units less than one record.     *
 * Otherwise to write one byte in a record a read/modify*
 * /write cyles would have to be done.                  *
 ********************************************************/

/********************************************************
 * Routine to read a value from a certain record.       *
 ********************************************************/

int
get_val( 
#if NeedFunctionPrototype
	struct sdb_file    *sp,
	int                recno, 
	int                pos, 
	union val_16_32    *val,
	int                precision)
#else
      sp, recno, pos, val, precision)
      struct sdb_file    *sp;         /* File to  access */
      int                recno;       /* Record number within file (starting at 1) */
      int                pos;         /* Offset within record (in bytes) */
      union val_16_32    *val;        /* Place to put result */
      int                precision;   /* Either PRECISION_16 or PRECISION_32 */
#endif
{
  /* Find out if the data held in the record buffer is of the correct
     record, and is valid for the pos range required */
  if( recno != sp->sd_recno || pos < sp->sd_start ||
				pos >= sp->sd_start + sp->sd_len ) {
    /* Flush the old record */
    if(flush_buf( sp ))
      return 1;
    /* The above is not true - we have to read the record */
    sp->sd_start = 0;
    sp->sd_len = REC_LEN;
    sp->sd_dirty = 0;
    sp->sd_recno = recno;
    if(read_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, recno,
		sp->sd_buff.u_buff, 0, REC_LEN))
      return 1;
  }

  memcpy( &val->u_sint, &sp->sd_buff.u_buff[pos], precision);
  return 0;
}

/********************************************************
 * Routine to write a value into a record buffer.       *
 ********************************************************/

int put_val( 
#if NeedFunctionPrototype
	    struct sdb_file   *sp, 
	    int               recno,
	    int               pos,
	    union val_16_32   *val, 
	    int               precision) 
#else
      sp, recno, pos, val, precision)
      struct sdb_file   *sp;          /* File to  access */
      int               recno;        /* Record number within file (starting at 1) */
      int               pos;          /* Offset within record (in bytes) */
      union val_16_32   *val;         /* Place to put result */
      int               precision;    /* Either PRECISION_16 or PRECISION_32 */
#endif
{
  char tmp_buf[REC_LEN];

  /* See if the record we have in the buffer is correct */
  if( recno != sp->sd_recno ) {
    /* Nope - flush it */
    if( flush_buf( sp ))
      return 1;
    /* Now clear out the record buffer and set the current
       record number for this buffer */
    memset( sp->sd_buff.u_buff, '\0', REC_LEN);
    sp->sd_recno = recno; /* Now looking at required record */
    sp->sd_start = sp->sd_len = 0; /* No bytes valid */
    sp->sd_dirty = 0; /* Not yet dirty */
  }

  /* Now we must put the value into the record buffer, setting
     the values of the valid start and length bytes correctly */
  if( sp->sd_len == 0 ) {
    /* First value to be written into this record buffer */
    sp->sd_start = pos;
    sp->sd_len = precision; /* Either 2 or 4 */
    sp->sd_dirty = 1; /* We now need to be written out */
  } else {
    /* Buffer has already been written into. If the next value to be
       written is not consecutive with the value already written then
       assume the record is going to be accessed in a random mannor.
       In that case it's much faster just to read the record off disk,
       modify any bytes within it, and then write the whole thing back
       out. */
    if( pos < (sp->sd_start - precision) || pos > (sp->sd_start+ sp->sd_len)) {
      /* Copy what we have into a temporary buffer, read the record, then
	 copy what we have back into the correct place */
      memcpy( tmp_buf, &sp->sd_buff.u_buff[sp->sd_start], sp->sd_len);
      if(read_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, recno,
		  sp->sd_buff.u_buff, 0, REC_LEN))
	return 1;
      memcpy( &sp->sd_buff.u_buff[sp->sd_start], tmp_buf, sp->sd_len);
      sp->sd_start = 0;
      sp->sd_len = REC_LEN;
      sp->sd_dirty = 1;
      sp->sd_recno = recno;
    } else {
      /* We are writing adjacent to an already written area */
      if( pos < sp->sd_start )
	sp->sd_start = pos;
      else
	sp->sd_len += precision;
    }
  }

  memcpy( &sp->sd_buff.u_buff[pos], &val->u_sint, precision);
  return 0;
}

/********************************************************
 * Routine to flush a record buffer belonging to a sp   *
 * structure. Does nothing if not dirty.                *
 ********************************************************/

int flush_buf( 
#if NeedFunctionPrototype
	      struct sdb_file *sp)
#else
      sp )
      struct sdb_file    *sp;    /* File to  access */
#endif
{
  /* Only flush buffer if it is dirty */
  if( !sp->sd_dirty )
    return 0;
  /* Else write as much as is dirty out */
  if(write_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, sp->sd_recno,
	       sp->sd_buff.u_buff, sp->sd_start, sp->sd_len))
    return 1;
  sp->sd_dirty = 0;
  return 0;
}

/********************************************************
 * Routine to return a validated pointer to a sdb_file  *
 * structure.                                           *
 ********************************************************/

struct sdb_file *get_sdb( 
#if NeedFunctionPrototype
			 int id,
			 int *retcode)
#else
      id, retcode)
      int    id;
      int    *retcode;
#endif
{
  if( id < 1 || id > NUM_FILES) {
    *retcode = INVALID_STREAM_ID;
    return NULL;
  }
  if( sd_files[id-1].sd_fd == 0 ) {
    *retcode = INVALID_STREAM_ID;
    return NULL;
  }

  return &sd_files[id-1];
}


/* List of error codes that can be returned */
static char *er_list[] = {
  "file not found",
  "unknown file id",
  "too many files in use simultaneously",
  "all streams in use",
  "file has faulty structure",
  "file is not an sdb file",
  "filename too long",
  "faulty request block", /* NEVER USED */
  "filing system error",
  "data management error", /* NEVER USED */
  "bad parameter in request",
  "point coordinates outside domain of interest",
  "file component does not exist",
  "record size too big or too small",
  "file already exists",
  "",
  "",
  "",
  "",
  "unable to encode calibration data",
  "unable to decode calibration data",
  "unsuported function",
  "already processing ipm on this route", /* NEVER USED */
  "insufficient access to user segment", /* NEVER USED */
  "request block not in user segment", /* NEVER USED */
  "service failure" /* NEVER USED */
  };
			    
/********************************************************
 * Routine to log any errors in the sdb routines.       *
 ********************************************************/

void log_error(
#if NeedFunctionPrototype
	       char   *name,
	       int    errcode)
#else
      name, errcode)
      char    *name;     /* Module error occured in */
      int     errcode;   /* sdb error code (defined in sdb.h) */
#endif
{

  /* Write out which module it failed in */
  fprintf( stderr, "%s: %s\n", name, er_list[errcode-1]);
  if( errno )
    perror( name );
}

/********************************************************
 * Routines that do reading and writing of fixed length *
 * records.                                             *
 * Deals with all the problems of writing to GEC files. *
 * blocksize as passed includes the block header length,*
 * recordsize does not.                                 *
 ********************************************************/

int rw_rec(
#if NeedFunctionPrototype
	   char   *name,
	   int    (*func)(),
	   int    fd,
	   int    blocksize,
	   int    recsize,
	   int    recno,
	   char   *buffer,
	   int    startpos,
	   int    len)
#else
      name, func, fd, blocksize, recsize, recno, buffer, startpos, len)
      char   *name;           /* Name of calling function */
      int    (*func)();       /* Either read or write */
      int    fd;              /* File to read/write */
      int    blocksize;       /* Block size on GEC (0 on unix) */
      int    recsize;         /* Record size we are reading/writing */
      int    recno;           /* Offset of record in file (start=1) */
      char   *buffer;         /* Buffer to read/write into */
      int    startpos;        /* Index into record we should read/write from */
      int    len;             /* Amount of record starting at startpos we should read/write */
#endif
{
  int recs_per_block;
  int blocks, records;
/*  long pos; */

  if( (blocksize && (recsize+4 > blocksize-4))||startpos > recsize|| len==0 ) {
    fprintf( stderr, "%s : parameters incorrect\n", name);
    return 1;
  }

  if( blocksize == 0 ) {
    /* We're on a UNIX filesystem - no worries about block or record
       headers */
    if( lseek( fd, (recno - 1)*recsize + startpos, L_SET) == -1) {
      fprintf( stderr, "%s seek fail\n", name);
      perror( name );
      return 1;
    }
  } else {
    /* We are on a GEC system - we must find the correct place */
    recs_per_block = (blocksize-4) / (recsize + 4);
    blocks = (recno - 1) / recs_per_block;
    records = (recno - 1) % recs_per_block;
    if( lseek( fd,((blocks * blocksize) + 4) + ((records * (recsize + 4)) + 4)+
	      startpos, L_SET) == -1) {
      fprintf( stderr, "%s seek fail\n", name);
      perror( name );
      return 1;
    }
  }

  /* Read the required number of bytes */
  if( (*func)( fd, &buffer[startpos], len)!= len) {
    fprintf( stderr, "%s read/write fail\n", name);
    perror( name );
    return 1;
  }
  return 0;
}

/********************************************************
 * Routine that calls the above to read an area of a    *
 * record.                                              *
 ********************************************************/

int read_rec(
#if NeedFunctionPrototype
	     int    fd,
	     int    blocksize,
	     int    recsize,
	     int    recno,
	     void   *buffer,
	     int    startpos,
	     int    len)
#else
      fd, blocksize, recsize, recno, buffer, startpos, len)
      int    fd;            /* File to read */
      int    blocksize;     /* Block size on GEC (0 on unix) */
      int    recsize;       /* Record size we are reading */
      int    recno;         /* Offset of record in file (start=1) */
      char   *buffer;       /* Buffer to read into */
      int    startpos;      /* Index into record we should read from */
      int    len;           /* Amount of record starting at startpos we should read */
#endif
{
#ifndef SVR4
      extern int read();
#endif

  return rw_rec( "read_rec :", read, fd, blocksize, recsize,
		recno, buffer, startpos, len);
}

/********************************************************
 * Routine that calls the above to write to an area of a*
 * record.                                              *
 ********************************************************/

int write_rec(
#if NeedFunctionPrototype
	      int    fd,
	      int    blocksize,
	      int    recsize,
	      int    recno,
	      void   *buffer,
	      int    startpos,
	      int    len)
#else
      fd, blocksize, recsize, recno, buffer, startpos, len)
      int     fd;           /* File to write */
      int     blocksize;    /* Block size on GEC (0 on unix) */
      int     recsize;      /* Record size we are writing */
      int     recno;        /* Offset of record in file (start=1) */
      char    *buffer;      /* Buffer to write from */
      int     startpos;     /* Index into record we should write to */
      int     len;          /* Amount of record starting at startpos we should write */
#endif
{
#ifndef SVR4
  extern int write();
#endif

  return rw_rec( "write_rec :", write, fd, blocksize, recsize,
		recno, buffer, startpos, len);
}

void ar_seekfail(
#if NeedFunctionPrototype
		 void
#endif
		 )
{
  fputs( "add_rec : seek fail\n", stderr);
  perror( "add_rec : ");
}

void ar_writefail(
#if NeedFunctionPrototype
		 void
#endif
		  )
{
  fputs( "add_rec : write fail\n", stderr);
  perror( "add_rec : ");
}

/********************************************************
 * Routine to add a record to a fixed length record file*
 * This routine has to generate block and record headers*
 * when writing to a GEC filesystem.                    *
 ********************************************************/

int add_rec(
#if NeedFunctionPrototype
	    int    fd,
	    int    blocksize,
	    int    recsize,
	    int    recno,
	    void   *buf)
#else  
      fd, blocksize, recsize, recno, buf)
      int     fd;           /* File to add to */
      int     blocksize;    /* Blocksize of file (0 on unix filesystem) */
      int     recsize;      /* Fixed length of a record */
      int     recno;        /* Record number to write (1==first) */
      char    *buf;         /* Record to write */
#endif
{
#ifndef SVR4
      extern char *calloc();
#endif
  short val = 0;
  int val32;
  int blocks, records, recs_per_block;

  if( blocksize == 0 ) {
    /* Unix filesystem - no problems */
    if(lseek( fd, (recno - 1) * recsize, L_SET)==-1) {
      ar_seekfail();
      return 1;
    }
  } else {
    /* GEC filesystem - this is where it gets tricky */
    recs_per_block = (blocksize-4) / (recsize + 4);
    blocks = (recno - 1) / recs_per_block;
    records = (recno - 1) % recs_per_block;
    if( records == 0 ) {
      /* This is the first record in a block - thus we must write
	 out the space for a new block */
      /* Seek to one byte before the end of the new block */
      if(lseek( fd, ((blocks+1) * blocksize)-1, L_SET)==-1) {
	ar_seekfail();
	return 1;
      }
      /* Now write a byte of zero to cause the file to be expanded */
      if(write(fd, &val, 1)!= 1) {
	ar_writefail();
	return 1;
      }
    }

    /* Now add the record to the file. First add the record to the
       length at the start of the block. */
    if( lseek(fd, blocks * blocksize, L_SET)==-1) {
      ar_seekfail();
      return 1;
    }

    /* Set the length pointer in the block to the correct record */
    val = ((records+1) * (recsize+4)) + 4;
    if(write(fd, &val, 2)!=2) {
      ar_writefail();
      return 1;
    }

    /* Now seek to the correct place to write out the record header */
    if(lseek(fd,((blocks*blocksize)+4) + (records*(recsize + 4)),L_SET)==-1) {
      ar_seekfail();
      return 1;
    }

    /* Write out the 4 byte record header */
    val32 = ((int)(recsize+4))<<16;
    if(write(fd, &val32, 4)!= 4) {
      ar_writefail();
      return 1;
    }
  }

  /* Finally write out the record itself */
  if(write(fd, buf, recsize)!=recsize) {
    ar_writefail();
    return 1;
  }

  return 0;
}

/********************************************************
 * Routine that uses GEC-NFS mechanism to tell the GEC  *
 * how many records a file has.                         *
 ********************************************************/

int set_numrecs( 
#if NeedFunctionPrototype
		int fd,
		int numrecs)
#else
      fd, numrecs)
      int fd;           /* File to use */
      int numrecs;      /* Number of records we are to set */
#endif
{
  numrecs |= 0xFFFF0000; /* This depends on 32 bit integers */
  return ftruncate( fd, numrecs);
}



/********************************************************
 * Routine to create a new spectrum file.               *
 ********************************************************/

int
sdef(
#if NeedFunctionPrototype
     int   *id,
     int   xbase,
     int   xrange,
     int   ybase,
     int   yrange,
     int   precision,
     char  *name, 
     int   namelen)
#else
      id, xbase, xrange, ybase, yrange, precision, name, namelen)
      int *id, xbase, xrange, ybase, yrange, precision, namelen;
      char *name;
#endif
{
  register struct sdb_file *sp;
  register struct template *tp;
  register int i;
  int retcode = 0;

  /* Validate the arguments */
  if( xrange <= 0 || yrange < 0 || (precision != PRECISION_16 &&
				       precision != PRECISION_32)) {
    log_error( "SDEF", retcode = BAD_ARGUMENTS);
    return(retcode);
  }

  /* Find a sdb_file structure to use */
  for( sp = &sd_files[0]; sp - &sd_files[0] < NUM_FILES; sp++)
    if( sp->sd_fd == 0)
      break;
  if( sp == &sd_files[NUM_FILES] ) {
    log_error( "SDEF", retcode = NO_MORE_STREAMS);
    return(retcode);
  }

  if( namelen > NAME_LENGTH - 1 ) {
    /* Limit on namelength in structure */
    log_error( "SDEF", retcode = NAME_TOO_LONG);
    return(retcode);
  }

  for( i = 0; (i < namelen) && name[i] != ' '; i++)
    sp->sd_name[i] = name[i];
  /* Null terminate the name string */
  sp->sd_name[i] = '\0';

  *id = (sp - &sd_files[0]) + 1; /* Stream id to return(retcode) */

  /* Try to open the file - fail if the file exists */
  if(access( sp->sd_name, F_OK)==0) {
    /* File exists */
    log_error( "SDEF", retcode = FILE_EXISTS);
    return(retcode);
  }

  /* Try and create the file properly */
  if((sp->sd_fd = open( sp->sd_name, O_CREAT|O_RDWR, 0644))==-1) {
    log_error( "SDEF", retcode = OPEN_FAIL);
    return(retcode);
  }

  /* Get the block size if we are on the GEC */
  sp->sd_blocksize = 0;

  /* Set the record buffer as unused */
  sp->sd_recno = 0;
  sp->sd_dirty = 0;

  /* Now we must set up the template record - first clear it to NULLS */
  memset( tp = &sp->sd_tmpl, 0, REC_LEN);
  /* Now set it up with the supplied values */
  
  tp->tm_norecs = 1; /* Only one record so far (this one !) */
  /* It's a historgram until I hear different.. */
  tp->tm_typecode = HISTOGRAM_SPECT; 
  tp->tm_revno = SDB_VERSION;
  tp->tm_dimension = ( yrange == 0 ) ? 1 : 2;
  tp->tm_precision = precision;
  tp->tm_date = (int) time(NULL); 
  tp->tm_xbase = xbase;
  tp->tm_xrange = xrange;
  tp->tm_ybase =  ybase;
  tp->tm_yrange = yrange;

  /* Now write out the first record */
  if(add_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, 1, tp)) {
    log_error( "SDEF", retcode = WRITE_ERROR);
    return(retcode);
  }

  return(retcode);
}


/********************************************************
 * Routine to return spectrum attributes, given an id.  *
 ********************************************************/

int
sinq( 
#if NeedFunctionPrototype
     int   id,
     int   *xbase,
     int   *xrange,
     int   *ybase,
     int   *yrange,
     int   *precision)
#else
      id, xbase, xrange, ybase, yrange, precision)
      int   id, *xbase, *xrange, *ybase, *yrange, *precision;
#endif
{
  struct sdb_file *sp;
  struct template *tp;
  int	 retcode = 0;

  /* Get id structure */
  if(( sp = get_sdb( id, &retcode))==NULL) {
    log_error( "SINQ", retcode);
    return(retcode);
  }

  tp = &sp->sd_tmpl;

  *xbase = tp->tm_xbase;
  *xrange = tp->tm_xrange;
  *ybase = tp->tm_ybase;
  *yrange = tp->tm_yrange;
  *precision = tp->tm_precision;
  
 return(retcode);
}


/********************************************************
 * Routine to open a spectrum fileand return(retcode) an id to   *
 * use in accessing it with these SDB routines.         *
 * If this code is called with namelen set to zero it   *
 * should release the file (ie. close it ).             *
 ********************************************************/

int
sname(
#if NeedFunctionPrototype
      int	*id,
      char	*name,
      int	namelen)
#else
      id,name,namelen)
      int	*id;
      char	*name;
      int	namelen;
#endif
{
  register int i;
  register struct sdb_file *sp;
  int retcode = 0; /* Set return(retcode)code as default OK */

  /* Check if we are to release a stream (or all streams) */
  if( namelen == 0 ) {
    if( *id == 0 ) {
      /* Close all streams */
      for( i = 0; i < NUM_FILES ; i++) {
	if( sd_files[i].sd_fd ) {
	  /* Flush the record buffer if neccessary */
	  flush_buf( &sd_files[i] );
	  if(sd_files[i].sd_blocksize) /*Tell GEC how many records there are */
	    set_numrecs( sd_files[i].sd_fd, sd_files[i].sd_tmpl.tm_norecs);
	  close( sd_files[i].sd_fd );
	  sd_files[i].sd_fd = 0; /* Mark stream as unused */
	}
      }
      return(retcode); /* We are finished */
    } else {
      /* Only close the stream passed in *id */
      if((sp = get_sdb( *id, &retcode))== NULL) {
	log_error( "SNAME", retcode);
	return(retcode);
      }
      flush_buf( sp );
      if(sp->sd_blocksize) /*Tell GEC how many records there are */
	    set_numrecs( sp->sd_fd, sp->sd_tmpl.tm_norecs);
      close( sp->sd_fd );
      sp->sd_fd = 0;
      return(retcode);
    }
  }

  /* Now find an empty stream and set it up */
  for( i = 0; i < NUM_FILES; i++)
    if( sd_files[i].sd_fd == 0 )
      break;
  if( i == NUM_FILES ) {
    log_error( "SNAME", retcode = NO_MORE_STREAMS);
    return(retcode);
  }
  
  if( namelen > NAME_LENGTH - 1 ) {
    /* Limit on namelength in structure */
    log_error( "SNAME", retcode = NAME_TOO_LONG);
    return(retcode);
  }

  *id = i + 1; /* Must be a +ve integer in id */
  sp = &sd_files[i];
  for( i = 0; (i < namelen) && name[i] != ' '; i++)
    sp->sd_name[i] = name[i];
  /* Null terminate the name string */
  sp->sd_name[i] = '\0';

  /* Now try and open the filename given */
  if(( sp->sd_fd = open( sp->sd_name, O_RDWR))==-1) {
    sp->sd_fd = 0;
    /* Extra info will be in errno */
    log_error( "SNAME", retcode = FILE_NOT_OPENED); 
    return(retcode);
  }

  sp->sd_blocksize = 0;

  /* Set the record buffer as unused */
  sp->sd_recno = 0;
  sp->sd_dirty = 0;
  
  /* Finally read in the first record (the template record) */
  if( read_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, 1, 
	       &sp->sd_tmpl, 0, REC_LEN)) {
    log_error( "SNAME", retcode = TEMPLATE_READ_ERROR);
    return(retcode);
  }
  return(retcode);
}


/********************************************************
 * Routine to read a range of counts from a spectrum.   *
 ********************************************************/

int
sread( 
#if NeedFunctionPrototype
      int   id,
      int   xbase,
      int   xrange,
      int   ybase,
      int   yrange, 
      int   *array,
      int   precision)
#else
      id, xbase, xrange, ybase, yrange, array, precision)
      int id, xbase, xrange, ybase, yrange, precision;
      int *array;
#endif
{
  struct sdb_file *sp;
  struct template *tp;
  register int lx_base, lx_range, ly_base, ly_range, num_to_read;
  int recno;
  register int pos;
  union val_16_32 val;
  short *spa = (short *)array; /* In case we have to read as short int16's */
  int tval;
  int retcode = 0;

  if( precision != PRECISION_16 && precision != PRECISION_32 ) {
    log_error( "SREAD", retcode = BAD_ARGUMENTS);
    return(retcode);
  }

  /* Get id structure */
  if(( sp = get_sdb( id, &retcode))==NULL) {
    log_error( "SREAD", retcode);
    return(retcode);
  }

  tp = &sp->sd_tmpl;

  /* Check the parameters passed */
  if((xbase < tp->tm_xbase) || 
     (xbase + xrange > tp->tm_xbase + tp->tm_xrange)) {
    log_error( "SREAD", retcode = BAD_ARGUMENTS);
    return(retcode);
  } else {
    lx_base = xbase;
    lx_range = xrange;
  }

  if( tp->tm_dimension == 2) {
    if((ybase < tp->tm_ybase) || 
       (ybase + yrange > tp->tm_ybase + tp->tm_yrange )) {
      log_error( "SREAD", retcode = BAD_ARGUMENTS);
      return(retcode);
    }
    ly_base = ybase;
    ly_range = yrange;
  } else {
    ly_range = 1;
    ly_base = 0;
  }

  /* Calculate the number of elements we are to read */
  for( num_to_read = ly_range * lx_range; num_to_read; ly_base++) {
    /* Get the offset in records (recno ==1 is first record in file)
       into the spectrum represented by lx_base, ly_base */
    recno = tp->tm_arrayrecs[gl_currentspec] +
            ((lx_base + (ly_base * tp->tm_xrange)) * tp->tm_precision)/REC_LEN;
    /* Calculate the offset in bytes into the record we need to read from */
    pos = ((lx_base + (ly_base * tp->tm_xrange)) * tp->tm_precision) % REC_LEN;

    /* Read xrange number of counts, then increment ybase */
    for( lx_range = xrange; lx_range; --lx_range) {
      /* Copy the number from the buffer into a temp value */
      get_val( sp, recno, pos, &val, tp->tm_precision);

      /* Read the correct value into a int precision variable */
      tval = tp->tm_precision == PRECISION_16 ? val.u_sshort : val.u_sint ;

      /* Now put it into the correct precision in the array */
      if( precision == PRECISION_16 ) {
	*spa++ = tval;
      } else {
	*array++  = tval;
      }
      --num_to_read;
      if( num_to_read && ((pos += tp->tm_precision) >= REC_LEN )) {
	/* We need to go to the next record number */
	recno++;
	pos -= REC_LEN; /* Put pos back within the buffer range */
      } /* End of "if( num_to_read && (pos += tp->tm_precision >= REC_LEN ))"*/
    } /* End of "for( lx_range = xrange; lx_range; --lx_range)" */
  } /* End of outer for */
return(retcode);
}


/********************************************************
 * Routine to write out an array of spectrum counts.    *
 ********************************************************/

int
swrit( 
#if NeedFunctionPrototype
      int   id,
      int   xbase,
      int   xrange,
      int   ybase,
      int   yrange, 
      int   *array,
      int   precision)
#else
      id, xbase, xrange, ybase, yrange, array, precision)
      int id, xbase, xrange, ybase, yrange, precision;
      int *array;
#endif
{
  struct sdb_file *sp;
  struct template *tp;
  int nrecs, num_to_write, recno;
  int lx_base, ly_base, lx_range, ly_range;
  union val_16_32 val;
  short *spa = (short *)array; /* In case we have to write as short int16's */
  long pos;
  char rec[REC_LEN];
  int retcode = 0;
  
  memset( rec, '\0', REC_LEN); /* Make sure we're writing zeros to new recs */

  if( precision != PRECISION_16 && precision != PRECISION_32 ) {
    log_error( "SWRITE", retcode = BAD_ARGUMENTS);
    return(retcode);
  }

  /* Get id structure */
  if(( sp = get_sdb( id, &retcode))==NULL) {
    log_error( "SWRITE", retcode);
    return(retcode);
  }

  tp = &sp->sd_tmpl;

  /* Check the parameters passed */
  if((xbase < tp->tm_xbase) || 
     (xbase + xrange > tp->tm_xbase + tp->tm_xrange)) {
    log_error( "SWRITE", retcode = BAD_ARGUMENTS);
    return(retcode);
  } else {
    lx_base = xbase;
    lx_range = xrange;
  }

  if( tp->tm_dimension == 2) {
    if((ybase < tp->tm_ybase) || 
       (ybase + yrange > tp->tm_ybase + tp->tm_yrange )) {
      log_error( "SWRITE", retcode = BAD_ARGUMENTS);
      return(retcode);
    }
    ly_base = ybase;
    ly_range = yrange;
  } else {
    ly_range = 1;
    ly_base = 0;
  }

  /* Now the hard part. We must allocate the neccessary records and
     update the template record to correspond with what we have written */

  if( tp->tm_arrayrecs[gl_currentspec] == 0 ) {
    /* No records allocated for this file */
    /* Calculate space for the whole file */
    nrecs = tp->tm_xrange * (tp->tm_yrange ? tp->tm_yrange : 1) *
      tp->tm_precision;
    /* Force it to number of records */
    nrecs = ( nrecs / REC_LEN) + ((nrecs % REC_LEN) ? 1 : 0 );

    /* Now create it by writing out the required number of records */
    for( pos = 0; pos < nrecs; pos++) {
      if(add_rec(sp->sd_fd, sp->sd_blocksize, REC_LEN, 
		 pos + tp->tm_norecs + 1,rec)) {/*1st record =1*/
	log_error( "SWRITE", retcode = SEEK_ERROR);
	return(retcode);
      }
    }

    /* If we did all this ok then set the new record length in template,
       and the base of this spectra in the correct array position, and
       update the template */
    tp->tm_arrayrecs[gl_currentspec] = tp->tm_norecs + 1;
    tp->tm_norecs += nrecs;

    /* Now update the template record */
    if(update_tmp( sp, &retcode))
      return(retcode);
  }

  /* Now we can write out the array given */
  /* Calculate the number of elements we are to write */
  for( num_to_write = ly_range * lx_range; num_to_write; ly_base++) {
    /* Get the offset in records (recno ==1 is first record in file)
       into the spectrum represented by lx_base, ly_base */
    recno = tp->tm_arrayrecs[gl_currentspec] +
            ((lx_base + (ly_base * tp->tm_xrange)) * tp->tm_precision)/REC_LEN;
    /* Calculate the offset in bytes into the record we need to write to */
    pos = ((lx_base + (ly_base * tp->tm_xrange)) * tp->tm_precision) % REC_LEN;

    /* Write xrange number of counts, then increment ybase */
    for( lx_range = xrange; lx_range; --lx_range) {
      /* Get the count in the correct precision from the array */
      if( precision == PRECISION_16 ) {
	val.u_sshort = *spa++;
      } else {
	val.u_sint = *array++;
      }

      /* Now put it into the buffer in the correct precision */
      put_val( sp, recno, pos, &val, tp->tm_precision);
      --num_to_write;
      if( num_to_write && ((pos += tp->tm_precision) >= REC_LEN )) {
	/* We need to go to the next record number */
	recno++;
	pos -= REC_LEN; /* Put pos back within the buffer range */
      }
    }
  }
return(retcode);
}


/********************************************************
 * Routine to update the template record after changes  *
 ********************************************************/

int update_tmp( 
#if NeedFunctionPrototype
	       struct sdb_file  *sp,
	       int              *retcode)
#else
      sp, retcode)
      struct sdb_file    *sp;          /* File to update */
      int                *retcode;
#endif
{
  struct template *tp = &sp->sd_tmpl;

  if(write_rec( sp->sd_fd, sp->sd_blocksize, REC_LEN, 1, tp, 0, REC_LEN)) {
    log_error( "Updating template record", *retcode = WRITE_ERROR);
    return 1;
  }
  return 0;
}

